'------------------------------------------------------------------
' AVR ESR Meter
' Author: Andrew Woodfield
' Date: July 2012
' File Name: esrmeter.bas

'------------------------------------------------------------------
' Program Description:

' A Tiny26 is used to measure the ESR of a capacitor using a
' constant current pulse and the internal 10-bit A/D converter, and
' over-sampling is used to improve the measurement dynamic range.
' The result is displayed on LEDs. Short circuits are also detected
' by the meter. The ESR meter automatically powers down after about
' 45 to 60 seconds.

'------------------------------------------------------------------
' Compiler Directives (some fundamental hardware issues)

$regfile = "attiny26.dat"     'tells bascom which device is being used
$crystal = 8000000            'bascom needs to know how fast it is going

'------------------------------------------------------------------
' Hardware Setups
' set up direction of all ports

Config Porta = Output         'make these micro pins outputs
Config Portb = Output         'set all output first, then inputs
Config Pinb.5 = Input         'input uses pin not port (for d/a)


' Hardware Aliases:

Vcc Alias Portb.4             'hardware alias for power switch
Led9 Alias Portb.6            'led to indicate a short circuit
Pulse Alias Portb.3           'current source for the test

' Initialise ports so hardware starts correctly
' ...Done in code below...

'------------------------------------------------------------------
' Declare Variables

Dim Count As Byte             '8 bit variables
Dim Temp As Byte
Dim Timeout As Byte           'simple counter to turn off esr meter
Dim Esr As Word               'result from 10 bit a/d (16 bit variable)
Dim Summed As Word            'used to average and filter the result

' Initialise Variables
' ...Done in code below...

' Declare Constants

Const Flashdelay = 75         'delay used several times (ms)

'------------------------------------------------------------------
' Program starts here

   Portb = 0                  'start from a known condition
   Set Vcc                    'the program has started so keep power on

'brief splash through the leds to show everything is working...

   Reset Led9                 'show the power is on with the red led
   Temp = &B01111111          'then turn each green led on in turn

   For Count = 1 To 8
     Porta = Temp             'write to port
     Waitms Flashdelay        'wait a preset time
     Rotate Temp , Right
   Next Count

'now start the actual esr measurement and display

For Timeout = 1 To 150        'simple method to determine power off time

   Porta = &B11111111         'turn leds off at start of each pass
   Set Led9                   'including this one

   For Count = 1 To 16        'employ some averaging (2 bit oversampling)
                               'this part is time-critical so it's back to asm
     $asm
     ldi r16,$85
     Out Adcsr , R16
     ldi r16,$88
     Out Admux , R16

     sbi portb,3              'start current source

     ldi r16,$4               '2uS delay
  Vsd:
     dec r16
     tst r16                  'zero?
     brne  vsd                'no, so continue very short delay
     sbi adcsr,adsc           'yes, so start conversion

     ldi r16,$12              '8uS delay for a/d to read
  Ssd:
     dec r16
     tst r16                  'zero?
     brne  ssd                'no, so continue somewhat short delay
     cbi portb,3              'yes, so stop current source
     'method relies on s/h getting value 1.5 adc cycles after start of conversion

  Readloop:
     sbis adcsr,adif          'wait until done
     rjmp readloop
     sbi adcsr,adif           'clear ready flag
     $end Asm

     Temp = Adcl              'use temp so adcl is read first
     Esr = Adch * 256
     Esr = Esr + Temp

     Summed = Summed + Esr

     Waitms 1                 'delay between samples
   Next Count

   Shift Summed , Right , 4   'divide by 16 gives oversampled result

   Select Case Summed         'case appears to assume a byte variable
      Case Is > 96 : Porta = &B11111111       'turn off top LED
      Case Is > 90 : Porta = &B11111110       'turn on top LED
      Case Is > 75 : Porta = &B11111101
      Case Is > 51 : Porta = &B11111011
      Case Is > 24 : Porta = &B11110111
      Case Is > 9 : Porta = &B11101111
      Case Is > 3 : Porta = &B11011111
      Case Is > 1 : Porta = &B10111111
      Case Is = 1 : Porta = &B01111111
      Case Is = 0 : Porta = &B11111111
      Case Else : Porta = &B11111111       'else turn off leds
   End Select

   If Summed = 0 Then         'if zero, test if large capacitor
     Set Pulse                'turn on current source
     Waitms 100
   'read a/d - this time using bascom
     Config Adc = Single , Prescaler = 32 , Reference = Internal
     Start Adc
     Esr = Getadc(8)          'read esr voltage (Range 0 to 1023)
     Reset Pulse              'turn current source off again

     If Esr > 79 Then         '80=200mV
       Porta = &B01111111     'must be large capacitor (Good 470uF - avg 10000uF)
       Set Led9               'so display accordingly
     Else
       Porta = &B11111111     'no, so must be a real short
       Reset Led9             'so indicate that it is a short circuit
     End If
   End If

   Waitms 300                 'wait 300mS to see final resulting display
                              ' and to allow large capacitors to discharge, if present
   Summed = 0                 'zero the summed variable for next time

Next Timeout                  ' go back to repeat until time to turn off

Reset Vcc                     'turn it off
Reset Vcc                     'repeat in case it takes more than one clock cycle
Reset Vcc                     'yes, I really want to turn the power off...
Waitms 500                    'I really mean it! And so...finally...the...


End

