;****************************************************************************************
;*   This file is the firmware to implement a LED and AC lamp switch, based on 		*
;*   the PIC12F683.  It monitors the zero crossing of the AC waveform,    the ambient   *
;*   light, the battery voltage and the flicking of a PIR sensor
;*   and drives a relay to switch the mains light or the LED light     *
;*                                                                			*
;*                                                                     			*
;*   Control is accomplished through one or more interruptions of power to the circuit,	*
;*   which causes a certain sequence of effects                		             	*
;*          			                                                        *
;*             light	   duration  interrupt   timeout                                *
;*    LDR senses "dark"       	       within      goto		                        *
;*					goto            				*             
;*    c0:      LED	   forever      c1        loop                          	*
;*    c1:      MAINS        5 sec  	c0	  c2                   	                *
;*    c2:      MAINS       10 sec       c4        c3               			*
;*    c3:      MAINS	   15 min 	c0        c0                			*
;*    c4:      MAINS	  120 min	c0        c0           				*
;*    		                        						*
;*											*
;*                                   							*
;*    "When it is dark, you will switch on LEDs, by interrrupting shortly you switch    *
;*     on MAINS for fifteen minutes, by waiting about ten seconds and interrupting      *
;*     then, you switch on MAINS 2 hours. If you interrupt a second time       		*
;*     without waiting, you will toggle between MAINS and LED light"                    *
;*          			                                                        *
;*             light	   duration  interrupt   timeout                                *
;*    LDR senses "dim or bright"       within      goto		                        *
;*					goto                         			*
;*    6:       MAINS        5 sec       c0   	   c7					*
;*    7:       MAINS       10 sec       c9         c8       				*
;*    8:       MAINS      120 min       c0         c0    				*
;*    9:       MAINS       15 sec       c0         c10					*
;*    10:      DIM LED     15 sec	c0         until dark:c11			*
;*    11:      MAINS     180-240 min    c0         c12					*
;*    12       DIM LED    120 min	c0         c13					*
;*    13:      DIM LED	   15 sec	c0	   until bright:c10			*
;*                                               					*
;*    "When it is not very dark, you will switch on then MAINS for 120 minutes. If	*
;*     you wait for 10 seconds and interrupt shortly, then lamp will go into 		*
;*    'goodnight' (burglar deterrent) mode. If you interrupt without waiting, you will 	*
;*     toggle between MAINS and LED mode. 'Goodnight' mode will wait until it becomes 	*
;*     dark, switch on the MAINS light and wait for a random time of 3 to 4 hours	*
;*     until it switches off."                                                          *
;*     											*
;*    "If you are unsure what mode the lamp is in, switch off for ten seconds to 	*
;*     restart"										*
;*											*
;*   GPIO 0 LDR sensor                   						*
;*   GPIO 1 relay output driver via transistor            				*
;*   GPIO 2 LED output driver via transistor                                 		*
;*   GPIO 3 zero cross detection input                   			        *
;*   GPIO 4 battery voltage sensor                      			        *
;*   GPIO 3 PIR detection input                   			        	*
;*                                                                     			*
;*   Refer to the MPASM User's Guide for additional information on     			*
;*   features of the assembler (Document DS33014).                     			*
;*                                                                     			*
;*   Refer to the respective PICmicro data sheet for additional        			*
;*   information on the instruction set.                               			*
;*                                                                     			*
;****************************************************************************************	
;*                                                                 			*
;*    Filename:	    trs683.asm                              		                *
;*    Date:         OCtober, 2009                                    			*
;*    File Version: A, this is a new file                                    		*
;*                                                                     			*
;*    Author:       G Michael Drygas                                      		*
;*                  Applications Engineer                    				*
;*    Company:      MICHAELS Unternehmergesellschaft(haftungsbeschrnkt)                       				*
;*                                                                     			*
;****************************************************************************************
;*                                			                                *
;*    Files required:   macros3b.asm                                          		*
;*                                                                     			*
;*                                                                     			*
;*                                                                     			*
;****************************************************************************************
;*                                                                     			*
;*    Notes:        hardware trafo 2t.brd                                                   			*
;*                               			*
;*                                                                     			*
;*                  matching \5\  .brd            		*
;*                                                                     			*
;****************************************************************************************


	list      p=12F683	; list directive to define processor
	#include  macros3b.asm
	config683B
	
#define OPTIONinit	B'11000101';	; clock 1:64 , WDT 1:1 ,	
#define TRISIOinit 	b'00111001'	; GPIO1,2 as OUTPUT, GPIO 0,3,4,5 as INPUT
#define GPIOinit 	b'00000110'	; GPIO1,2 HIGH
#define ANSELinit 	B'01111000' 	; ADC Frc clock, ((AN0,))AN3(GP4) as analog
#define	ADCON0init	b'00001101'	; left justified , AN3, on

#define PR2init		.128
#define CCPR1Linit      PR2init - .10
#define CCP1CONinit	b'00000000'	; disable pwm
#define T2CONinit       b'00000100'	; switch on TMR2

;***** RESET VECTOR AND START OF CODE

	ORG     0x000           ; start coding 
                            	nop
	btfss	STATUS,NOT_TO	; a WDT-Timeout occurred
	 goto	afterwdt
	goto PWRUP
	ORG	0x004		;interrupts code
	return			; disable interrupts


#define Lo_Intensity	.255
#define Hi_Intensity	.20
	
vtable	addwf	PCL,F
	dt	2* (1+1)-1
	dt 0x32,Lo_Intensity
	dt 0xB6,Hi_Intensity

	; 11.9V	10110110   $B6
	; 12.6  10100101   $A5
	; 13.6  10001110   $8C
	; 14.1   1000011   $43	




undoreset
	cblock	_MINRAM			; start of variables

	    SystemStatus		; byte that holds vital flags
	    CasE
	    DeadCount,IsDead
	    TimerCounter:3,TimerVector
	    AL
	    BatAvr:2
	endc
	#define STACK	_MAXRAM

;	init_andis			; initiation analog data display
;	eq_	FSR,=,BatAvr,	lit8	; variable to display in indirect register

	init683				; initiation

	return
;******************************************************************************
;***** CONSTANT DEFINITIONS

#define		Active_Flag	SystemStatus,7
#define		After_Pwrup	SystemStatus,6
#define		AC_is_ON	SystemStatus,5
#define		AC_is_High	SystemStatus,4
;#define		Twice_Pass	SystemStatus,3
;#define		twicebit	b'00001000'
;*****************T W I C E ****************************************
;#define		TWCES		0
;*****************************************************************************
#define	TIrun		SystemStatus,3
#define	Read_Out	SystemStatus,2
#define	After_Godead	SystemStatus,1
#define	MAINS_On	SystemStatus,0
	;#define	mainsbit	b'00000001'

;******************************************************************************

#define 	LDR_Port	GPIO,0
#define 	LED_Port	GPIO,2
#define 	PIR_Port	GPIO,5
#define 	Relay_Port	GPIO,1

#define 	Z_Detect_Port	GPIO,3		; zero crossing input for Reset on Change

#define		pat0	b'01010101'	;11	; pattern switching
#define		pat1	b'00110011'	;10
#define		pat2	b'01110100'	;01
#define		pat3	b'00111001'     ;00
#define		pat4	b'00011110'


;******************************************************************************

settimer	macro	time,unit,MaxTmr
		  variable	optn,intime,k,clow,chigh,chi2,chi3,dus
k=1
	if unit == msec
k=.1000
	endif
	if unit == sec
k=.1000000
	endif


	IFDEF	__12F683
	
		; OSCON:
				;bit 6-4 IRCF<2:0>: Internl Oscillator Frequency Select bits
				;000 = 31 kHz
				;001 = 125 kHz
				;010 = 250 kHz
				;011 = 500 kHz
				;100 = 1 MHz
				;101 = 2 MHz
				;110 = 4 MHz
				;111 = 8 MHz
	IFDEF	OSCONinit
optn =	OSCONinit & b'01110000'
	ELSE
optn =	b'01100001'
	ENDIF

optn = optn >> 4
	
	
	if	optn == 0
dus = 65
	else
dus = 1<<(7-optn)
	endif
	ELSE
dus = 2
	ENDIF
	
	
optn = 0
 	IFDEF	OPTIONinit	
	 if ! (OPTIONinit & b'00001000')
optn = 1 + (OPTIONinit & b'00000111')
	endif	
	ENDIF
; OSCONinit * OPTIONinit

dus = dus << optn		; prescale 


intime = time *k * 2 / dus 
					; intime = intime + 65336
clow =	intime	& 0xFF
chigh=  (intime >> 8) &0xFF
chi2  =  (intime >> .16)& 0xFF
chi3  =  (intime >> .24)
	if chi3 > .255
	 
	 error "Required Delay needs higher clock prescaler !!"	 
	 
	else

	if	clow != 0
chigh+=1
	 if	chigh == .256
chi2 +=1
chigh==0
	 endif
	endif
	if chigh !=0
	 if	chi2 == .256
chi3 +=1
chi2=.255
	 endif
chi2 +=1
chi2 = chi2 & 0xFF	
	endif
	if intime < .1024
	movlw	0x00FF - (intime & 0xFF)
	movwf	TMR0
	endif
	if (chi2 < 5) & (chi3 == 0)
	 eq_ 	MaxTmr,=,chigh,		lit8	
	endif

	 eq_	MaxTmr+1,=,chi2,	lit8
	 eq_	MaxTmr+2,=,chi3+1,	lit8
	endif
	endm


PWRUP		call	undoreset	
		clrf	SystemStatus		; clr all flags incl Mains_On
		call	batADC
		eq_	BatAvr,=,ADRESH,byte
		movlw	0
		call	case_W_6		; case 0 or 6  to start with


gomain		bsf	After_Pwrup		; wait 18*3 msec
		eq	DeadCount,.3		; for sufficient voltage
		goto	gsleep
				
godead		eq	DeadCount,.4		; wait 72ms to retest AC line
		bsf 	After_Godead


;		settimer	.120,sec,TimerCounter;  2 minutes timertrap

testACin	clrwdt
		call	acon		; if not AC_On		
		btfss	AC_is_ON
		 goto 	tstLDR
				
		if_bit	TIrun
		 call case6

		 goto	gomain		

tstLDR		btfsc	LDR_Port	; if	PIR_on and dark enough
		 goto	nobatt		; WARNING LDR not triggered by LED light

		if_	BatAvr,>, 0xBF,		lit8
		 goto	nobatt	;	if BatAver > 11.8V
;
		btfss	PIR_Port	;	15 minutes timertrap
		 goto inloop
		settimer	.15*.60,sec,TimerCounter
		bsf	TIrun 

;		eq_	AL,=,20,lit8
;cloop		 call	battery
;		blue	5
;		clrwdt
;		call rblink
;		decfsz	AL,f
;		goto cloop
;
	
;		blue	5
;		clrwdt
;		goto $-1
;
		call linearze

;		blue	5
;		clrwdt
;		goto $-1
;


inloop		if_not	TIrun
		 goto nobatt	
		if_not	INTCON,T0IF
		 goto	testACin
		bcf	INTCON,T0IF	
		decfsz	TimerCounter,F
	 	 goto	testACin

	 	call	battery 		; update battery
		call	linearze
		decfsz	TimerCounter+1,F
	 	 goto   testACin		; expired counter
 
nobatt		movlw	b'11110011'	; disable pwm
		andwf	CCP1CON,F
		bsf	GPIO,2	        ;make sure transistor is HIGH
		bcf	TIrun
;		red 5
;		clrwdt
;		goto $-1

		incfsz	IsDead,F
		 goto   gsleep

		clrf	CasE			; after 7.2sec start with LED
		bcf	MAINS_On
		goto	gsleep	
gsleep		sleep
;******************************************************************************
afterwdt 	decfsz	DeadCount,F		; 
		 goto	gsleep				; wait for DeadCount*18ms
		call	battery 		; update battery
		btfss	After_Pwrup		; 
		 goto	Not_pwrup

		call undoreset
	
		goto MAIN

Not_pwrup	btfss	After_Godead		; loop until AC is on
		 goto 	switchedoff		
		goto 	godead

switchedoff	

		eq	IsDead,.155
		call	poll11msAC		; check if AC is really dead
		btfsc	AC_is_ON
		 goto	gomain			; resume

	#ifdef TWCES			; switch only
		movlw	twicebit		;
		xorwf	SystemStatus,F		; every second pass
		btfsc	Twice_Pass		;
		 goto	godead 
	#endif
		
		movfw	CasE
		addwf	PCL,F
		 goto	makecase1		; 0
		 goto	makecase0		; 1
		 goto	makecase4		; 2
		 goto	makecase0		; 3	
		 goto	makecase0		; 4
		 goto	makecase0		; 5
		 goto	makecase0		; 6
		 goto	makecase9		; 7
		 goto	makecase0		; 8
		 goto	makecase0		; 9
		 goto	makecase0		;10
		 goto	makecase0		;11
		 goto	makecase0		;12
		 goto	makecase0		;13


makecase0	eq	CasE,.0
		bcf	MAINS_On
;		clrf	TimerCounter+2		;test
		;call	set5sectime		;test
;		bcf	Dim_LED
		goto	godead

makecase1	call	set5sectime
		movlw	.1
		call	case_W_6
		bsf	MAINS_On
		goto	godead




makecase4	eq	CasE,.4
		bsf	MAINS_On
		call	set120mintime
		goto	godead		

makecase9	eq	CasE,.9
		call	set15sectime
		goto	godead		

	
	
nextPIN		;bcf	STATUS,7		;simulator error???
		clrwdt
;		movlw	ledbit
;		btfsc	Dim_LED
;		 xorwf	GPIO,F	
;	 	sleep				; wait for PIN wakeup			


;******************************************************************************

afterpin 	btfss	After_Pwrup
 		 goto	notfirst

MAIN		bcf	After_Pwrup
		bcf	After_Godead
		clrf	TMR0			; 16.384 msec
		eq 	DeadCount, .3		; max16+54 ms timeout


;*************************************************************test
;		call	flic
;*************************************************************test		

notfirst	clrwdt
		call acon	
		if_bit	AC_is_ON
		 goto active
		if_not	INTCON,T0IF
		 goto	afterpin
		if_not	Active_Flag
		 goto	pwrlost 


;		call acon
;		if_bit	AC_is_ON	
;		 goto	blu
;		red 5
;		clrwdt
;		goto notfirst
;blu		blue 5
;		
;		clrwdt
;		decfsz	STACK,F
;		 goto $-2
;		goto notfirst
;

		bcf	INTCON,T0IF
		bcf	Active_Flag
	
	
;		anadis 5, blue,red,black	; analog data display, GPIO5,
						; long=high, short=low, off=hi Z
		goto	afterpin	

pwrlost		bsf 	Relay_Port		; switch off relay
						; pwm
		movlw	b'11110011'	; disable pwm
		andwf	CCP1CON,F
		bsf	GPIO,2	        ;make sure transistor is HIGH
;		red 5
		goto	gsleep

active			
;		blue 5
;		clrwdt
;		decfsz	STACK,F
;		 goto $-2
		clrwdt	
		bsf	Active_Flag
		decfsz	TimerCounter,F
		 goto	apdistr

		decfsz	TimerCounter+1,F
		 goto	apbatt				;2.55 sec /50Hz
		decfsz	TimerCounter+2,F
		 goto	apdistr

;*************************************************************test
;		call	flic
;*************************************************************test		




timedout	movfw	CasE
		addwf	PCL,F
		 goto	tocase0		; case 0->> loop		
		 goto	tocase2		;1
		 goto	tocase3         ;2
		 goto	tocase0         ;3
		 goto	tocase0         ;4
		 goto	tocase0         ;5
		 goto	tocase7         ;6
		 goto	tocase8         ;7
		 goto	tocase0      	;8
		 goto	tocase10	;9
		 goto	loopuntildark   ;10
		 goto	tocase12   	;11
		 goto	untilbright     ;12
		 goto	tocase10      	;13
	
tocase0		clrf	CasE
		bcf	MAINS_On
;		call	set10sectime	;test
;		bcf	Dim_LED
                goto	nextPIN

tocase2		eq	CasE,.2
		call	set10sectime
;		bsf	MAINS_On		
		goto	nextPIN

tocase3		eq	CasE,.3
		call	set15mintime
;		bsf	MAINS_On		
		goto	nextPIN

tocase4		eq	CasE,.4
		call	set120mintime
;		bsf	MAINS_On		
		goto	nextPIN

tocase7		eq	CasE,.7
		call	set10sectime
;		bsf	MAINS_On		
		goto	nextPIN

tocase8		eq	CasE,.8
		call	set120mintime
;		bsf	MAINS_On		
		goto	nextPIN

tocase10	eq	CasE,.10
t15		call	set15sectime
		bcf	MAINS_On
;		bsf	Dim_LED		
		goto	nextPIN


loopuntildark	incf	AL,F
		btfss	LDR_Port
		 goto	dark
		call	set15sectime
		goto	nextPIN
dark		eq	CasE,.11

		swapf	AL,W
		movwf	TimerCounter+1
		andlw	0x0F
		movwf	TimerCounter+2
		incl	TimerCounter+2,.16
;		call 	set120mintime		;test

		bsf	MAINS_On
		goto	nextPIN

tocase12	eq	CasE,.12
		call	set120mintime
		bcf	MAINS_On
;		bsf	Dim_LED		
		goto	nextPIN



untilbright	btfss	LDR_Port
		 goto	t15
bright		goto	tocase10


;******************************************************************************


set15mintime	eq_	TimerCounter+1,=,0x0101+ (.100 * .60 * .15 / .256),lit16; <15 minutes   	
		retlw	0

set120mintime	eq_	TimerCounter+1,=,0x0101+ (.100 * .60 * .120 / .256),lit16; <120 minutes   	
		retlw	0

set5sectime	eq_	TimerCounter,=,0x010101+ (.105 * .5 ),lit24; <5 seconds   	
		retlw	0

set15sectime
set10sectime	eq_	TimerCounter,=,0x010101+ (.105 * .10  ),lit24; <10 seconds   	
		retlw	0




acon		mov_bit	Z_Detect_Port, Read_Out	;	skpanotb Z, AC_H
		bsf	AC_is_ON		;	 goto nandb
		if__    Read_Out ,equal,  AC_is_High	;	movlw acbit|aconbit
	  	 bcf	AC_is_ON		;	xorwf Status
		mov_bit	Read_Out , AC_is_High	;	retlw
		retlw	0			;nandb	skpanotb AC_H,Z
						;	 goto acoff
						;	bsf	acon
						;	-bcf	acbit
						;	ret
						;acoff	bcf	acon
						;	ret
poll11msAC	eq	TMR0,.86
wakeloop	clrwdt
		call	acon
		btfsc	AC_is_ON
		 retlw	0	
		incf	TMR0,W
		skpZ
		 goto	wakeloop
		retlw	0

case_W_6	btfsc	LDR_Port		; detect pitch black LDR07/2.7MOhm
	 	 goto	case6
		movwf	CasE			
		goto	sett

case6		eq	CasE,.6
		bsf	MAINS_On
sett		call	set5sectime	
		retlw	0

;************************************** test
;flic		movfw	CasE
;		movwf	BHH
;		setc
;		rlf	BHH,F	;1..3..5
;		movlw	b'0000110'		 
;		xorwf	GPIO,F
;flicloop	clrwdt
;		decfsz	BL,F
;		 goto	$-2
;		decfsz	BH,F
;		 goto	$-4
;		xorwf	GPIO,F
;		decfsz	BHH,F
;		 goto	flicloop
;		retlw	0
;;************************************** test


batADC	
	;eq_	ADCON0,=,b'00001101',lit8	; left justified , AN3, on
	;dlay	6
	bsf	ADCON0,GO
	btfsc	ADCON0,GO
	 goto	$-1
	retlw	0

battery
		call	batADC
		clrf	BatAvr+1
		movfw	BatAvr
		addwf	BatAvr,F
		skpNC
		 incf	BatAvr+1,F
		addwf	BatAvr,F
		skpNC
		 incf	BatAvr+1,F
		movfw	ADRESH
		addwf	BatAvr,F
		skpNC
		 incf	BatAvr+1,F
		clrC
		rrf	BatAvr+1,F
		rrf	BatAvr,F
		clrC
		rrf	BatAvr+1,F
		rrf	BatAvr,F
	retlw 0		
		
apbatt		call	battery
				; if ambient light is broad daylight
				; switch off mains
				; if ambient light is brighter than LEDs
				; switch off LED
				; if battery voltage > 14.7
				; switch on LED regardless, declare overloaded
				; if overloaded and voltage > 13.8
				; switch on LED regardless
apdistr		
;		clrwdt
;		goto $-1	
		btfsc	MAINS_On
		 goto	relayon

		bsf	Relay_Port
		nop			;btfss	Dim_LED
ledon		call	linearze
		;	 bcf	LED_Port

	
		goto	nextPIN
	
relayon		movlw	b'11110011'	; disable pwm
		andwf	CCP1CON,F
		bsf	GPIO,2	        ;make sure transistor is HIGH
		nop
		bcf	Relay_Port		; switch on Relay;

		



;apdistr	btfsc	AC_is_High		; after PIN wakeup do 1firing cycle
;   	 	 goto start8plus		;
;		goto	start8minus
;
;
;
;
;
;;
;;	eq Q3wait,.40
;;	;eq	Q1Wait,.35	
;start8minus	
;	mov	StarTer,BL
;	goto	$+1
;	goto	$+1
;	goto	$+1
;	goto	$+1
;	decfsz	BL,F
;	 goto	$-5
;	decfsz	BH,F
;	 goto	nodec
;	eq	BH,.5	
;	decfsz	StarTer,F
;	 decf	StarTer,F
;	incf	StarTer,F	
;nodec	eq	IndeX,.10	;delay to zerocross
;	decfsz	IndeX,F
;	 goto $-1
;	
;;	skpneq	MinusT,MaxTmr
;;	 goto	nextPIN		; dont fire
;;	movf	MinusT,F
;;	skpnZ
;	 goto	fire		; fire now
;;	mov	TMR0,StartupCount
;;	inc	StartupCount,MinusT
;;	goto	timetoc
;
;start8plus	mov	StarTer,BL
;	goto	$+1
;	goto	$+1
;	goto	$+1
;	goto	$+1
;	decfsz	BL
;	 goto	$-5
;
;	eq	IndeX,.50	;delay to zerocross
;	decfsz	IndeX,F
;	 goto $-1
;;	skpneq	PlusT,MaxTmr
;;	 goto	nextPIN			; dont fire
;;	movf	PlusT,F
;;	skpnZ
;	 goto	fire		; fire now
;;	mov	TMR0,StartupCount
;;	inc	StartupCount,PlusT
;;timetoc	clrwdt	
;;	skpeq	TMR0,StartupCount
;;	 goto	timetoc
;fire	bcf	Relay_Port
;	clrwdt
;	eq	IndeX, 8
;	decfsz	IndeX,F		;1
;	 goto	$-1		;2,3 *8 =24us
;	bsf	Triac_Port

	goto	nextPIN		;



;#define	STACK	_MAXRAM
#define MdB	STACK-0
#define LoB	STACK-1
#define HiB	STACK-2

#define MdD	STACK-3
#define LoD	STACK-4
#define HiD	STACK-5
  
;#define	AL	BatAvr
;#define AL	x	;test
 
linearze	eq_	AL,=,BatAvr,	byte
		eq_	MdB,=,.1,	lit8
		call	vtable
		movwf	LoB
		if_	AL,>,LoB,	byte
		 goto	bases
exbdry		do_	MdB,=,MdB,+,1,	lit8
		movfw	MdB
		call	vtable
		movwf	MdD
		goto done

bases		movlw	0
		call	vtable
		movwf	MdB
		call	vtable
		movwf	HiB
		if_	AL,>=,HiB,	byte
		 goto exbdry

basesearch	do_	MdB,=,MdB,-,(1+1),lit8 
		movfw	MdB
		call	vtable
		movwf	LoB
		if_	AL,>=,LoB,	byte
		 goto 	binsearch
		eq_	HiB,=,LoB,	byte
		goto	basesearch

binsearch	do_	MdB,=,MdB,+,1,	lit8
		movfw	MdB
		call	vtable
		movwf	LoD
		do_	MdB,=,MdB,+,(1+1),	lit8
		movfw	MdB
		call	vtable
		movwf	HiD

binloop		do_	MdB,=,LoB,+,HiB,	(byte)
		rrf	MdB,F
		do_	MdD,=,LoD,+,HiD,	(byte)
		rrf	MdD,F
		
		if_	AL,>,MdB,		(byte)
		 goto	higher

		skpNZ				; AL equal MdB
		 goto	done

		eq_	HiB,=,MdB,	byte
		eq_	HiD,=, MdD,	byte
		goto	binloop
higher		eq_	LoB,=,MdB,	byte
		eq_	LoD,=,MdD,	byte
		goto	binloop

done		eq_	PR2,=,MdD,	byte	;set frequency
		do_	CCPR1L,=,MdD,-,.10,lit8 ; for PNP-transistor
	
		; switch on LED with linearizer
		; if	PIR_on 15 minutes timertrap
		;	else
		;	 step out this routine
		; endw

		; if AC goes on while no PIR = case 0/6
		;	else case 6
		movlw	b'00001100'	; enable pwm
		iorwf	CCP1CON,F
		return


rblink		clrwdt
		decfsz	STACK,F
		 goto $-2
		decfsz	STACK-1,F
		 goto $-4

		red	5
		clrwdt
		decfsz	STACK,F
		 goto $-2
		decfsz	STACK-1,F
		 goto $-4

	return


	end
