;***********************************************************************
;*   This file is the firmware to implement a AC lamp dimmer, based on *
;*   the PIC10F200.  It monitors the zero crossing of the AC waveform  *
;*   and pulses the gate of a TRIAC appropriately for dimming an AC    *
;*   lamp.                                                             *
;*                                                                     *
;*   Intenisty control is accomplished through two mechanisms, 1 is the*
;*   interruption of power to the circuit, which causes a time out of  *
;*   the watchdog timer, and 2 through an IR receiver.  A secondary    *
;*   circuit implementing the IR remote control is also included for   *
;*   the control side.  Finally, the circuit also implements a long    *
;*   term power off timeout to reset the IntensiTy to zero.  While the *
;*   circuit is running another long term timeout shuts down the       *
;*   circuit in the event that the light is left on for more than 4    *
;*   hours.                                                            *
;*                                                                     *
;*   GPIO 0 enable pin used to power the IR receiver                   *
;*   GPIO 1 data output from the IR receiver                           *
;*   GPIO 2 triac output driver                                        *
;*   GPIO 3 zero cross detection input the the circuit                 *
;*                                                                     *
;*   Refer to the MPASM User's Guide for additional information on     *
;*   features of the assembler (Document DS33014).                     *
;*                                                                     *
;*   Refer to the respective PICmicro data sheet for additional        *
;*   information on the instruction set.                               *
;*                                                                     *
;***********************************************************************
;*                                                                     *
;*    Filename:	    dimmer.asm                                         *
;*    Date:         July 28, 2004                                      *
;*    File Version: A.0                                                *
;*                                                                     *
;*    Author:       g michael                                      *
;*                  Principal Applications Engineer                    *
;*    Company:      Michaels Prototype                     *
;*                                                                     *
;***********************************************************************
;*                                                                     *
;*    Files required:                                                  *
;*                                                                     *
;*                                                                     *
;*                                                                     *
;***********************************************************************
;*                                                                     *
;*    Notes:                                                           *
;*                                                                     *
;*                                                                     *
;*                                                                     *
;*                                                                     *
;***********************************************************************

	#include p10f202.inc
	#include macros.inc
	__CONFIG _MCLRE_OFF & _CP_OFF & _WDT_OFF & _IntRC_OSC

	ERRORLEVEL -306				; Get rid of banking messages...

;******************************************************************************
;***** VARIABLE DEFINITIONS

	cblock 0x08
		IndeX				; Define all user varables starting at location 0x7
		SystemStatus			; byte that holds system flags
				; counts dead cycle to zero IntensiTy
		CounTer:3			; used for 4hr timer
				; holds IR low min time for decode
		CountsHalfCycle:2		; holds time delay for triac startup
		StartupCount:2			; temperary variable
		MaxTmr:2			; maximal value btween 2 Z_cross
	endc

;******************************************************************************
;***** CONSTANT DEFINITIONS

#define Tmr0_Bit7	SystemStatus,7
#define Test_Wdt	SystemStatus,6
#define Dun_Flag	SystemStatus,5			; interlock to prevent repeat of commands
#define Phase_Plus	SystemStatus,0			; used to determined if zerocross or IR data change

;***** 4 hour timer preset macro
four_hours	macro				; macro to load delay time for 4 hour timer
	movlw	0x00
	movwf	counter
	movlw	0x5E
	movwf	counter+1
	movlw	0x1A
	movwf	counter+2
	endm

;******************************************************************************
;***** RESET VECTOR AND START OF CODE
	ORG     0xFF            ; processor reset vector
	ORG     0x000           ; start coding 
	btfsc	STATUS,NOT_PD	; Test for the first power up
	 goto	PWRUP
	
	call	undoreset
	btfsc	STATUS,GPWUF	; a Wakeup on PinChange occurred
	 goto	afterpin

	btfss	STATUS,NOT_TO	; a WDT-Timeout occurred
	 goto	afterwdt
	; goto error
#define 	TRISGPIO 	6
#define 	Triac_Port	GPIO,2
#define Rx_Enable_Port	GPIO,0				; enables IR receiver
#define Rx_Data_Port	GPIO,1				; carries IR receiver data input
#define	Triac_Gate_Port	GPIO,2				; active low triac drive
#define Z_Detect_Port	GPIO,3				; zero crossing input for Reset on Change

undoreset			;movlw	B'00001000'	; set direction bits; GPIO3 is input (high-z) GPIO0:2 are output 
	movlw	b'11111010'	; configure Triac_Gate_Port and IR power as output
	TRIS 	TRISGPIO	
;	movlw	B'01000100'	; PIN wakeup enable, weak pullup disabled
				; Timer0 internal, clock 1:32 , WDT 1:1
				
	movlw	B'01001000'	; PIN wakeup enable, weak pullup disabled
				; Timer0 internal clock, clock 1:1 , WDT 1:1 ,
	OPTION	
	retlw	0


PWRUP	andlw	b'11111110'	;bit 0 FOSC4: INTOSC/4 Output Enable bit(1)
	;iorwl	b'00000001'
	movwf   OSCCAL          ; update register with factory cal value 

	movlw	b'00000000'	; set all Outputs to Low Voltage
	movlw	b'00000101'	; turn off the Triac_Gate_Port and IR receiver
	movwf	GPIO

	call	undoreset
	
;comp	movlw	b'11110111'     ; configure comparator inputs as digital I/O
;	movwf	CMCON0		; comment instruction for PIC10F200 /202
	

	goto	MAIN
					

;******************************************************************************
tmr0check	btfss	TMR0,7
		 goto	$+5
		btfsc	Tmr0_Bit7
		 retlw	0			; both set
		bsf	Tmr0_Bit7			;
		retlw 0
		btfss	Tmr0_Bit7
		 retlw 0			; both not set
		bcf	Tmr0_Bit7
		incf	MaxTmr+1,F		; change from high to low eg FF to 00
		retlw	0		

MAIN
	eq	IndeX,.10
wait10c	clrwdt
	btfss	Z_Detect_Port			; test frequency
	 goto	$-1
	btfsc	Z_Detect_Port			; test frequency
	 goto	$-1
	decfsz	IndeX,F
         goto	wait10c
	eeq	MaxTmr, 0
	clrwdt
	bcf	Tmr0_Bit7
	clrf	TMR0
	call	tmr0check
	btfss	Z_Detect_Port
       	 goto	$-2
	mov	TMR0, MaxTmr
	mov16	MaxTmr, CountsHalfCycle
	
startc	mov16	CountsHalfCycle, StartupCount
	shr16	StartupCount, 4			; div 16
	btfsc	Z_Detect_Port			; test cycle
	 goto	$-1
	clrf16	MaxTmr
	clrf	TMR0
	bsf	Triac_Port
uptime	call	tmr0check
	mov	TMR0, MaxTmr
	skp16ge	MaxTmr, StartupCount
	 goto uptime
	bcf	Triac_Port


	btfss	Z_Detect_Port			; could go sleep
	 goto	$-1				; neg cycle
	goto	startc				; ignore


	clrf16	MaxTmr
	clrf	TMR0
	bsf	Triac_Port
dntime	call	tmr0check
	mov	TMR0, MaxTmr
	skp16ge	MaxTmr, StartupCount
	 goto 	dntime
afterpin
afterwdt
	sleep
	end
