;***********************************************************************
;* Software License Agreement                                          *
;* The software supplied herewith by Microchip Technology Incorporated *
;* (the "Company") is intended and supplied to you, the Company's      *
;* customer, for use solely and exclusively on Microchip products.     *
;*                                                                     *
;* The software is owned by the Company and/or its supplier, and is    *
;* protected under applicable copyright laws. All rights are reserved. *
;* Any use in violation of the foregoing restrictions may subject the  *
;* user to criminal sanctions under applicable laws, as well as to     *
;* civil liability for the breach of the terms and conditions of this  *
;* license.                                                            *
;*                                                                     *
;* THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES,   *
;* WHETHER EXPRESS, IMPLIED OR STATU-TORY, INCLUDING, BUT NOT LIMITED  *
;* TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A         *
;* PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,   *
;* IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR          *
;* CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.                   *
;*                                                                     *
;***********************************************************************
;*   This file is the firmware to implement a AC lamp dimmer, based on *
;*   the PIC10F200.  It monitors the zero crossing of the AC waveform  *
;*   and pulses the gate of a TRIAC appropriately for dimming an AC    *
;*   lamp.                                                             *
;*                                                                     *
;*   Intenisty control is accomplished through two mechanisms, 1 is the*
;*   interruption of power to the circuit, which causes a time out of  *
;*   the watchdog timer, and 2 through an IR receiver.  A secondary    *
;*   circuit implementing the IR remote control is also included for   *
;*   the control side.  Finally, the circuit also implements a long    *
;*   term power off timeout to reset the IntensiTy to zero.  While the *
;*   circuit is running another long term timeout shuts down the       *
;*   circuit in the event that the light is left on for more than 4    *
;*   hours.                                                            *
;*                                                                     *
;*   GPIO 0 enable pin used to power the IR receiver                   *
;*   GPIO 1 data output from the IR receiver                           *
;*   GPIO 2 triac output driver                                        *
;*   GPIO 3 zero cross detection input the the circuit                 *
;*                                                                     *
;*   Refer to the MPASM User's Guide for additional information on     *
;*   features of the assembler (Document DS33014).                     *
;*                                                                     *
;*   Refer to the respective PICmicro data sheet for additional        *
;*   information on the instruction set.                               *
;*                                                                     *
;***********************************************************************
;*                                                                     *
;*    Filename:	    dimmer.asm                                         *
;*    Date:         July 28, 2004                                      *
;*    File Version: A.0                                                *
;*                                                                     *
;*    Author:       Keith Curtis                                       *
;*                  Principal Applications Engineer                    *
;*    Company:      Microchip Technology Incorporated                  *
;*                                                                     *
;***********************************************************************
;*                                                                     *
;*    Files required:                                                  *
;*                                                                     *
;*                                                                     *
;*                                                                     *
;***********************************************************************
;*                                                                     *
;*    Notes:                                                           *
;*                                                                     *
;*                                                                     *
;*                                                                     *
;*                                                                     *
;***********************************************************************

	#include p10f206.inc

	__CONFIG _MCLRE_OFF & _CP_OFF & _WDT_ON & _IntRC_OSC

	ERRORLEVEL -306				; Get rid of banking messages...

;******************************************************************************
;***** VARIABLE DEFINITIONS

	cblock 0x08				; Define all user varables starting at location 0x7
		SystemStatus			; byte that holds system flags
		IntensiTy			; 3 bit number for current IntensiTy setting
		DeadCount			; counts dead cycle to zero IntensiTy
		counter:3			; used for 4hr timer
		delay				; used for triac drive
		ir_counter			; counts time IR input is low
		dark_counter			; counts time IR input is high
		ir_minimum			; holds IR low min time for decode
		StartupCount:2				; holds time delay for triac startup
		hold				; temperary variable
	endc

;******************************************************************************
;***** CONSTANT DEFINITIONS

#define Rx_Enable_Port	GPIO,0				; enables IR receiver
#define Rx_Data_Port	GPIO,1				; carries IR receiver data input
#define	Triac_Gate_Port	GPIO,2				; active low triac drive
#define Z_Detect_Port	GPIO,3				; zero crossing input for Reset on Change

#define Dun_Flag	SystemStatus,7			; interlock to prevent repeat of commands
#define Phase_Plus	SystemStatus,0			; used to determined if zerocross or IR data change

;***** 4 hour timer preset macro
four_hours	macro				; macro to load delay time for 4 hour timer
	movlw	0x00
	movwf	counter
	movlw	0x5E
	movwf	counter+1
	movlw	0x1A
	movwf	counter+2
	endm

;******************************************************************************
;***** RESET VECTOR AND START OF CODE

	org 0x0					; Reset vector is at 0x0000
Main
	btfsc	STATUS,NOT_PD			; Test for the first power up
	 goto	PWRUP

	btfss	STATUS,NOT_TO			; Test for a watch dog reset (loss of 50/60Hz)
	 goto	PWRDOWN				; 18ms

	call	Init_pu
	btfsc	STATUS,GPWUF			; Test for a pin change reset (zero cross detect)
	 goto	Wakeup				; sort of WDT 10ms = 100/120Hz

	call	Init_v
wait
	nop
	sleep					; goto minimal current state

;******************************************************************************
;***** First Power up
PWRUP
	call	Init_pu
	clrwdt





	bsf	Dun_Flag

	movlw	0x08
	movwf	IntensiTy
	movlw	.15
	movwf	ir_minimum
	movlw	.16
	movwf	ir_counter

	movlw	0x80
	movwf	StartupCount+1
	clrf	StartupCount

	four_hours
	goto	wait

;******************************************************************************
;***** Zero cross reset (GPIO change)
Wakeup
	clrwdt
	btfss	Phase_Plus				; test for zero cross change (ignore if irdata change)
	 goto	low_Phase_Plus

hi_Phase_Plus					; if last test was high then input should be low
	btfsc	Z_Detect_Port
	 goto	wait				; if current and last are high then irdata change so ignore
	bcf	Phase_Plus				; if not then update Phase_Plus
	goto	zero_x				; and respond to zero cross detect

low_Phase_Plus					; if last test was low then input should be high
	btfss	Z_Detect_Port
	 goto	wait				; if current and last are low then irdata change so ignore
	bsf	Phase_Plus				; if not then update Phase_Plus
	goto	zero_x				; and respond to zero cross detect

zero_x
	movlw	.20				; reset the dead cycle counter
	movwf	DeadCount

get_delays
	bcf	STATUS,C
	rlf	IntensiTy,w
	movwf	hold
	btfss	Z_Detect_Port				; test for Phase_Plus of 60Hz
	 goto	second_Phase_Plus
first_Phase_Plus
	call	Table1_rd
	movwf	StartupCount+1
	incf	hold,w
	call	Table1_rd
	movwf	StartupCount
	goto	triac_drive
second_Phase_Plus
	call	Table2_rd
	movwf	StartupCount+1
	incf	hold,w
	call	Table2_rd
	movwf	StartupCount

triac_drive
	btfsc	StartupCount+1,7
	 goto	IR_decode		; no triac pulse on 0x80
	movf	StartupCount+1,f
	skpNZ
	 goto	triac_pulse

triac_loop
	decfsz	StartupCount,f	;1
	 goto	triac_loop	;2,3        ;StartupCount us*3 +
	decfsz	StartupCount+1,f	;768us* countH) + counth*3
	 goto	triac_loop

triac_pulse
	bcf	Triac_Gate_Port
pulse_delay
	decfsz	StartupCount,F
	 goto	pulse_delay	;256us
	bsf	Triac_Gate_Port

IR_decode
	btfsc	Rx_Data_Port				; if ir_data is high goto next
	 goto	ir_high

	movf	ir_counter,f			; if ir_counter == 0 goto end of decode
	skpNZ
	 goto	ir_end

	decf	ir_counter,w			; ir_counter--
	movwf	ir_counter
	movwf	ir_minimum			; ir_minimum   = ir_counter
	clrf	dark_counter			; dark_counter = 0
	goto	ir_end
ir_high
	movlw	.16
	movwf	ir_counter			; ir_counter = 16

	movlw	.31
	subwf	dark_counter,w
	skpC
	 incf	dark_counter,f			; if dark_counter < 31 then decrement

	movlw	.31
	xorwf	dark_counter,w
	skpZ			; if dark_counter <> 31 then ignore
	 goto	ir_end

	movlw	0xFF
	movwf	dark_counter			; dark_counter = 255

	movf	ir_minimum,f
	skpNZ
	 goto	command_end			; if ir_minimum !>0 then error

	movlw	.7
	subwf	ir_minimum,w
	skpNC
	 goto	mode_dec			; if ir_minimum > 6 then must be decrement
mode_inc

	decf	IntensiTy,f
	movlw	0x08
	btfsc	IntensiTy,7
	 movwf	IntensiTy
	movf	IntensiTy,w
	four_hours

	goto	command_end

mode_dec
	movlw	.13
	subwf	ir_minimum,w
	skpNC			; if ir_minimum > 13 then must be error
	 goto	command_end

	incf	IntensiTy,f
	movlw	0x09
	xorwf	IntensiTy,w
	skpNZ
	 clrf	IntensiTy
	movf	IntensiTy,w
	four_hours

command_end
	movlw	.16
	movwf	ir_minimum

ir_end

	bsf	Triac_Gate_Port
end_pulse
	bcf	Dun_Flag
	bcf	STATUS,GPWUF			; clear the flag
	decfsz	counter,f
	 goto	wait
	decfsz	counter+1,f
	 goto	wait
	decfsz	counter+2,f
	 goto	wait
	movlw	0x08
	movwf	IntensiTy
	goto	wait

;******************************************************************************
;***** Power interruption reset (WDT)
PWRDOWN
WDT_init
	call	Init_pd				; was a reset, configure peripherals
	clrwdt					; push out the dog timeout

	decfsz	DeadCount,f			; count the WDT time out
	 goto	cmd_test			; if not 50 then goto the command test
	clrf	IntensiTy
	goto	wait

cmd_test
	btfss	Dun_Flag				; test for first pass through wdt
	 goto	new_cmd				; if yes, then do the command
	goto	pwr_end

new_cmd
	bsf	Dun_Flag				; flag wdt command call as done
	decf	IntensiTy,f
	decf	IntensiTy,f
	movlw	0x08
	btfsc	IntensiTy,7
	 movwf	IntensiTy
	movf	IntensiTy,w
	four_hours
pwr_end
	movf	GPIO,w
	bcf	STATUS,GPWUF			; clear the flag
	goto	wait


;******************************************************************************
;***** SYSTEM SUBROUTINES
Init_pu
	movlw	b'11111010'			; configure Triac_Gate_Port and IR power as output
	tris	GPIO				; all others are input
	movlw	b'00000101'			; turn off the Triac_Gate_Port and IR receiver
	movwf	GPIO
	movlw	b'01011011'			; enable IOC, ps > wdt, Fosc/4 > TMR0
	OPTION					; wakeup,intclck, 1:8 WDT = 8*18ms = 144msec
	movlw	b'01100001'			; <4>intclck<3>cmp off
	movwf	CMCON0
	retlw	0

Init_pd
	movlw	b'00000100'			; turn off the Triac_Gate_Port, turn on IR receiver
	movwf	GPIO
	movlw	b'11111000'			; configure Triac_Gate_Port and IR power as output
	tris	GPIO				; all others are input
	movlw	b'01011100'			; enable IOC, ps > wdt, Fosc/4 > TMR0
	OPTION
	movlw	b'01100001'
	movwf	CMCON0
	retlw	0

Init_v
	bcf	Dun_Flag
	retlw	0

Table1_rd					; table for first half of waveform
	addwf	PCL,f
tab1_strt
	retlw	0x80				;   0.0%
	retlw	0x00
	retlw	0x07				;  16.6% 5592us of 8333;   6667 of 10000 
	retlw	0x45				;			   0xA6  0x08
	retlw	0x05				;  25.0%
	retlw	0x6D
	retlw	0x03				;  33.3%
	retlw	0x94
	retlw	0x00				;  50.0%
	retlw	0x00
	retlw	0x00				;  66.6%
	retlw	0x00
	retlw	0x00				;  75.0%
	retlw	0x00
	retlw	0x00				;  83.4%
	retlw	0x00
	retlw	0x00				; 100.0%
	retlw	0x00
Table2_rd					; table for second half of waveform
	addwf	PCL,f
tab2_strt
	retlw	0x80				;   0.0%
	retlw	0x00
	retlw	0x80				;  16.6%
	retlw	0x00
	retlw	0x80				;  25.0%
	retlw	0x00
	retlw	0x80				;  33.3%
	retlw	0x00
	retlw	0x80				;  50.0%
	retlw	0x00
	retlw	0x07				;  66.6%
	retlw	0x45
	retlw	0x05				;  75.0%
	retlw	0x6D
	retlw	0x03				;  83.4%
	retlw	0x94
	retlw	0x00				; 100.0%
	retlw	0x00

;******************************************************************************
;***** END OF CODE
	end
