unit Edt;

interface

const
  EdtX1 : byte = 1;
  EdtY1 : byte = 1;
  EdtX2 : byte = 78;
  EdtY2 : byte = 24;
  EdtTop : string = 'MEMO-Editor';

  procedure TextEdit (filename : string);

implementation

uses
  dos, crt, screen;

  function MaxX : byte;
  begin
    MaxX := succ(lo(WindMax) - lo(WindMin));
  end;

  function MaxY : byte;
  begin
    MaxY := succ(hi(WindMax) - hi(WindMin));
  end;

  procedure clreol;
  var
    x, y : byte;
  begin
    x := wherex;
    y := wherey;
    write('':succ(MaxX) - x);
    gotoxy(x, y);
  end;

  procedure NormVideo;
  begin
    TextAttr := $07;
  end;

  procedure HighVideo;
  begin
    NormVideo;
    TextAttr := TextAttr or $08;
  end;

  procedure InversVideo;
  begin
    NormVideo;
    TextAttr := TextAttr and $f8;
    TextAttr := TextAttr or $70;
  end;

  procedure BlinkVideo;
  begin
    TextAttr := TextAttr or $80;
  end;

(****************************************************************************)
(*                                                                          *)
(*   LiesZeichen mit Tastenconvertierung                                    *)
(*                                                                          *)
(****************************************************************************)

  procedure LiesZeichen(var C : char);
  const
    ScanCode = #0;
    Puffer : char = #0;
  begin
    if Puffer <> #0 then
      begin
        C := Puffer;
        Puffer := #0;
        exit;
      end;
    C := readkey;
    if C = ScanCode then
      begin
        C := readkey;
        case C of
          #72 :
            C := ^E; (* Cursor hoch *)
          #80 :
            C := ^X; (* Cursor tief *)
          #75 :
            C := ^S; (* Cursor links *)
          #77 :
            C := ^D; (* Cursor rechts *)
          #73 :
            C := ^R; (* Page up *)
          #81 :
            C := ^C; (* Page down *)
          #115 :
            C := ^A; (* Ctrl + Cursor links *)
          #116 :
            C := ^F; (* Ctrl + Cursor rechts *)
          #82 :
            C := ^V;
          #83 :
            C := ^G;
          #71 :
            begin
              C := ^Q;
              Puffer := ^S;
            end; (* Home *)
          #79 :
            begin
              C := ^Q;
              Puffer := ^D;
            end; (* End *)
          #132 :
            begin
              C := ^Q;
              Puffer := ^R;
            end; (* Ctrl + Page Up *)
          #118 :
            begin
              C := ^Q;
              Puffer := ^C;
            end; (* Ctrl + Page Down *)
          #119 :
            begin
              C := ^Q;
              Puffer := ^E;
            end; (* Ctrl + Home *)
          #117 :
            begin
              C := ^Q;
              Puffer := ^x;
            end; (* Ctrl + End *)
          else
            C := #0;
        end; (* Case *)
      end; (* if *)
  end; (* LiesZeichen IBM *)

(****************************************************************************)
(*                                                                          *)
(*   Implementiert dynamische Strings in Turbo-Pascal                       *)
(*                                                                          *)
(*   t StatString = STRING[255]                                             *)
(*   t DynString = dynamischer String                                       *)
(*   f DynStringLength : INTEGER  -> Laenge eines dynamischen Strings       *)
(*   p MakeDynString(StatString,DynString)                                  *)
(*       kopiert StatString in DynString                                    *)
(*       Achtung! Keine Speicherueberpruefung                               *)
(*   p MakeStatString(DynString,StatString)                                 *)
(*       kopiert DynString in StatString                                    *)
(*   p ForgetDynString(DynString)                                           *)
(*       gibt den von DynString belegten Speicherplatz wieder frei          *)
(*   f DynStringLess(X,Y,upper):BOOLEAN                                     *)
(*       X,Y : DynString                                                    *)
(*       upper : BOOLEAN                                                    *)
(*       -> TRUE, wenn X lexikalisch vor Y kommt, ansonsten FALSE           *)
(*       Falls upper=TRUE, wird Gross/Kleischrift ignoriert                 *)
(*   f DynStringEqual(X,Y,upper):BOOLEAN                                    *)
(*       Parameter wie bei DynStringLess                                    *)
(*       -> TRUE, wenn X gleich Y, ansonsten FALSE                          *)
(*   f DynStringPos(X,Y,Start,upper):INTEGER                                *)
(*       -> Liefert das erste Vorkommen von X in Y ab dem Startzeichen      *)
(*       upper wie bei DynStringLess                                        *)
(*                                                 Codegroesse ca. 1Kbyte   *)
(****************************************************************************)

procedure TextEdit (Filename : string);

type
  StatString = string[255];
  DynString = ^StatString;
const
  edited : boolean = false;
  function DynStringLength(D : DynString) : integer;
  begin
    DynStringLength := ord(D^[0])
  end; (* DynStringLength *)

  procedure MakeDynString(S : StatString; var D : DynString);
  var
    l : byte absolute S;
  begin
    getmem(D, succ(l));
    move(S, D^, succ(l))
  end; (* MakeDynString *)

  procedure MakeStatString(D : DynString; var S : StatString);
  begin
    S := D^
  end; (* MakeStatString *)

  procedure ForgetDynString(var D : DynString);
  begin
    freemem(D, succ(DynStringLength(D)))
  end; (* ForgetDynString *)

  function DynStringLess(x, y : DynString; upper : boolean) : boolean;
  var
    i, l1, l2, m : integer;
  begin
    if not upper then
      begin
        DynStringLess := x^ < y^;
        exit;
      end;
    l1 := DynStringLength(x);
    l2 := DynStringLength(y);
    i := 1;
    if l1 < l2 then
      m := l1
    else
      m := l2;
    while (i <= m) and (upcase(x^[i]) = upcase(y^[i])) do
      i := succ(i);
    if i <= m then
      DynStringLess := upcase(x^[i]) < upcase(y^[i])
    else
      DynStringLess := l1 < l2;
  end; (* DynStringLess *)

  function DynStringEQUAL(x, y : DynString; upper : boolean) : boolean;
  var
    i, l1, l2 : integer;
  begin
    if not upper then
      begin
        DynStringEQUAL := x^ = y^;
        exit;
      end;
    l1 := DynStringLength(x);
    l2 := DynStringLength(y);
    if l1 <> l2 then
      begin
        DynStringEQUAL := false;
        exit;
      end;
    for i := 1 to l1 do
      if upcase(x^[i]) <> upcase(y^[i]) then
        begin
          DynStringEQUAL := false;
          exit;
        end;
    DynStringEQUAL := true;
  end; (* DynStringEQUAL *)

  function DynStringPos(x, y : DynString; Start : integer; upper : boolean) : integer;
  var
    i, j, l1, l2 : integer;
  begin
    l1 := DynStringLength(x);
    l2 := DynStringLength(y);
    for i := Start to succ(l2 - l1) do
      begin
        j := 0;
        if upper then
          while (j < l1) and (upcase(x^[succ(j)]) = upcase(y^[i + j])) do
            j := succ(j)
        else
          while (j < l1) and (x^[succ(j)] = y^[i + j]) do
            j := succ(j);
        if j = l1 then
          begin
            DynStringPos := i;
            exit;
          end
      end;
    DynStringPos := 0;
  end; (* DynStringPos *)

(****************************************************************************)
(*  TED-1.INC   Typdefinition und Basisprozeduren fr den Turbo-Editor TED  *)
(****************************************************************************)

type
  TextListe = ^TextZeile;              (* TextListe ist eine doppelt *)
  TextZeile = record                   (* verkettete Liste           *)
                Zeile : DynString;
                Vor, Rueck : TextListe
              end;

  function HeapAvail : integer;
  var
    temp : longint;
  begin
    temp := maxavail;
    if temp > maxint then
      HeapAvail := maxint
    else
      HeapAvail := temp;
  end;

  procedure NeuerText(var T : TextListe);    (* Initialisierung *)
  begin
    T := nil;
  end;

  procedure LiesText(var f : text; var T : TextListe);
  var
    l : StatString;
    D : DynString;
    p, q : TextListe;
  begin
    NeuerText(T);
    reset(f);
    q := T;
    while not eof(f) do
      begin
        if HeapAvail < 1000 then
          begin
            close(f);
            exit;
          end;
        readln(f, l);
        new(p);
        with p^ do
          begin
            MakeDynString(l, Zeile);
            Rueck := q;
            Vor := nil;
          end;
        if q <> nil then
          q^.Vor := p
        else
          T := p;
        q := p
      end;
    close(f)
  end;

  procedure SchreibText(var f : text; T : TextListe);
  begin
    rewrite(f);
    while T <> nil do
      with T^ do
        begin
          writeln(f, Zeile^);
          T := Vor;
        end;
    close(f);
  end;

  procedure LoescheText(var T : TextListe);
  var
    p : TextListe;
  begin
    while T <> nil do
      begin
        p := T^.Vor;
        ForgetDynString(T^.Zeile);
        dispose(T);
        T := p;
      end;
  end;

var
  T : TextListe;
  f : text;
  x, y, Z : integer;
  C : char;
{  S : string[255];
 }
(****************************************************************************)
(*    TEDINST.INC  Installation von TED (Abgestimmt auf Herkules-Karte)     *)
(****************************************************************************)

type
  VideoString = string[80];

var
  ActiveColor : byte;

  procedure SetTextColor;
  begin
    ActiveColor := 15
  end;

  procedure SetBlockColor;
  begin
    ActiveColor := 112
  end;

  procedure SetStatusColor;
  begin
    ActiveColor := 7
  end;

  procedure MoveToScreen(S : VideoString; x, y : integer);
  type
    Screentyp = array[1..24, 1..80] of word;
  var
    i : integer;
    OldAttr, oldactive : byte;
    ActScreen : ^Screentyp;
  begin
    if DirectVideo then
      begin
        x:=pred(x);
        if LastMode = Mono then
          ActScreen:=ptr($b000,0)
        else
          ActScreen:=ptr($b800,0);
        oldactive := activecolor;
        for i:=1 to length(S) do
         begin
         if (s[i]='-') and (s[i+1] in ['1'..'9']) then
        activecolor := lightred;
      if (s[i]='-') and (s[i+1] = '0') then
        activecolor := red;

      if (s[i]='+') and (s[i+1] in ['1'..'9']) then
        activecolor := lightgreen;
      if (s[i]='+') and (s[i+1] = '0') then
        activecolor := green;
      if (s[i]=^G) then
        activecolor := yellow;

      if s[i]=' ' then
        activecolor := oldactive;

          if S[i]>=#32 then
            ActScreen^[y+edty1,edtx1+x+i]:=byte(S[i])+ActiveColor shl 8
          else
            ActScreen^[y+edty1,edtx1+x+i]:=64+byte(S[i])+1 shl 8;
         {   if s[i] =^G then
  begin
    sound(1000);
    delay(300);
    nosound;
  end;    }
         end;
         activecolor :=white;
      end
    else
    begin
      OldAttr := TextAttr;
      TextAttr := ActiveColor;
      gotoxy(x, y);
      write(S);
      TextAttr := OldAttr;
    end;
  end; (* MoveToScreen *)

(****************************************************************************)
(*  TED-2.INC  Der erste Teile der Prozedur EditText fr den Turbo-Editor   *)
(****************************************************************************)

  procedure EditText(var T : TextListe; x, y, Z : integer;
                     Spalten, ZeilenGanz : integer; Status : boolean);
  const
    MaxSpalten = 255;
  var
    Zeile : StatString; (* Die Zeile, innerhalb der ediert wird. *)
    l : byte absolute Zeile; (* Die ZeilenLnge *)
    temp, Anfang : TextListe; (* Der Zeiger auf letzte Zeile *)
    C : char; (* Das eingegebene Zeichen *)
    LR, RO : integer; (* Linker und oberer Rand *)
    Zeilen : integer; (* Die nutzbaren Zeilen *)

  const
    Buchstaben : set of char = ['A'..'Z', 'a'..'z', '0'..'9'];
    NaechsterBefehl : char = #0;
    InsertMode : boolean = true;
    IndentMode : boolean = true;
    SpaltenBlock : boolean = false;

  type BlockMarke = record
                      Line : TextListe;
                      tx, tz : integer;
                      Markiert : boolean;
                    end;

  var
    BlockAnfang, BlockEnde : BlockMarke;

    procedure AppendLine(var T : TextListe; Z : DynString);
    var
      temp : TextListe;
    begin
      new(temp);
      with temp^ do
        begin
          Zeile := Z;
          Rueck := T;
          if T <> nil then
            begin
              Vor := T^.Vor;
              T^.Vor := temp;
              if Vor <> nil then
                Vor^.Rueck := temp;
            end
          else
            Vor := nil;
        end; (* WITH *)
      T := temp;
    end; (* InsertLine *)

    procedure DeleteLine(var T : TextListe);
    begin
      if T <> nil then 
        begin
          with T^ do
            begin
              ForgetDynString(Zeile);
              if Vor <> nil then
                Vor^.Rueck := Rueck;
              if Rueck <> nil then
                Rueck^.Vor := Vor;
              dispose(T);
              T := nil;
            end; (* WITH *)
        end; (* IF *)
    end; (* DeleteLine *)

    function Copies(C : char; N : integer) : StatString;
    var
      S : StatString;
    begin
      fillchar(S[1], N, C);
      S[0] := chr(N);
      Copies := S;
    end; (* Copies *)

    procedure StatusZeile;
    var
      i : byte;
    begin
      gotoxy(1, 1);
      NormVideo;
      write ('':MaxX);
      i := length(filename);
      while (i > 0) and (filename[i] <> '\') do
        i := pred(i);
      if i > 0 then
        delete(filename, 1, i);
      for i := 1 to length(filename) do
        filename[i] := upcase(filename[i]);
      gotoxy(5, 1);
      write(filename);
      gotoxy(20, 1);
      write(' Zl:      Sp:     St:');
      if InsertMode then
        write(' Einf.');
      if IndentMode then
        write(' Einr.');
      HighVideo;
    end;

    procedure SchreibStatus;
    begin
      gotoxy(24, 1);
      write(Z:5);
      gotoxy(34, 1);
      write(x:3);
    end;

    procedure ChangeMode;
    begin
      InsertMode := not InsertMode;
      if Status then
        begin
          StatusZeile;
          SchreibStatus;
        end;
    end;

    procedure ChangeIndent;
    begin
      IndentMode := not IndentMode;
      if Status then
        begin
          StatusZeile;
          SchreibStatus;
        end;
    end;

    procedure Print(var S : StatString; x, y, Sp : integer);
    var
      T : string[80];
      lt : byte absolute T;
      ls : byte absolute S;
      xp, nt, ns : integer;
    begin
      if Sp < 1 then
        exit;
      if Sp > Spalten then
        Sp := Spalten;
      if x <= LR then
        x := succ(LR);
      xp := x - LR; if xp > Sp then
        exit;
      nt := succ(Sp - xp);
      ns := succ(ls - x);
      if nt > ns then
        nt := ns;
      T := copy(S, x, nt);
      nt := succ(Sp - lt - xp);
      if nt > 0 then
        begin
          fillchar(T[succ(lt)], nt, ' ');
          lt := lt + nt;
        end;
      MoveToScreen(T, xp, y + RO);
    end; (* print *)

    procedure ShowLine(var S : StatString; x, y, Z : integer);
    begin
      if not(BlockAnfang.Markiert and BlockEnde.Markiert) or
      (Z < BlockAnfang.tz) or (Z > BlockEnde.tz) then
        Print(S, x, y, Spalten)
      else if ((Z = BlockAnfang.tz) and (Z = BlockEnde.tz)) or SpaltenBlock then
        begin
          Print(S, x, y, pred(BlockAnfang.tx - LR));
          SetBlockColor;
          Print(S, BlockAnfang.tx, y, pred(BlockEnde.tx - LR));
          SetTextColor;
          Print(S, BlockEnde.tx, y, Spalten);
        end
      else if Z = BlockAnfang.tz then
        begin
          Print(S, x, y, pred(BlockAnfang.tx - LR));
          SetBlockColor;
          Print(S, BlockAnfang.tx, y, Spalten);
          SetTextColor;
        end
      else if Z = BlockEnde.tz then
        begin
          SetBlockColor;
          Print(S, x, y, pred(BlockEnde.tx - LR));
          SetTextColor;
          Print(S, BlockEnde.tx, y, Spalten)
        end
      else
        begin
          SetBlockColor;
          Print(S, x, y, Spalten);
          SetTextColor;
        end;
    end; (* ShowLine *)

    procedure ShowPage(T : TextListe; ab : integer);
    var
      i, iz : integer;
      LeerZeile : string[80];
    begin
      iz := Z + ab - y;
      LeerZeile := Copies(' ', Spalten);
      for i := ab to succ(Zeilen) do
        begin
          if NaechsterBefehl <> #0 then
            if KeyPressed then
              begin
                LiesZeichen(C);
                if C = NaechsterBefehl then
                  exit;
              end;
          if T = nil then
            MoveToScreen(LeerZeile, 1, i + RO)
          else
            begin
              ShowLine(T^.Zeile^, succ(LR), i, iz);
              iz := succ(iz);
              T := T^.Vor
            end;
        end;
      C := #0;
    end; (* ShowPage *)

    procedure AktualisiereBlocks(cx, cz : integer);
    var
      xp : integer;
      procedure Pruefe(cx, cz : integer; var Block : BlockMarke);
      begin
        with Block do
          if Markiert then
            begin
              if Z < tz then
                tz := tz + cz
              else if Z = tz then
                begin
                  if cz < 0 then
                    tx := 1
                  else if (xp > tx + cx) and (xp <= tx) then
                    tx := xp
                  else if xp < tx then
                    tx := tx + cx;
                end;
            end;
      end;

    begin
      xp := x + LR;
      Pruefe(cx, cz, BlockAnfang);
      Pruefe(cx, cz, BlockEnde);
    end; (* AktualisiereBlocks *)

    procedure aktualisiere;
    var
      D : DynString;
      p : TextListe;
    begin
      while Zeile[l] = ' ' do
        l := pred(l);
      if Zeile = temp^.Zeile^ then
        exit;
      MakeDynString(Zeile, D);
      AppendLine(temp, D);
      p := temp^.Rueck;
      if T = p then
        T := temp;
      DeleteLine(p);
    end; (* aktualisiere *)

    function Top : TextListe;
    var
      i : integer;
      p : TextListe;
    begin
      p := temp;
      for i := 2 to y
        do p := p^.Rueck;
      Top := p;
    end; (* Top *)

    procedure HochSchieben;
    var
      p : TextListe;
    begin
      aktualisiere;
      p := Top;
      NaechsterBefehl := ^Z;
      if temp^.Vor = nil then
        exit;
      repeat
        if y > 1 then
          y := pred(y)
        else
          begin
            Z := succ(Z);
            temp := temp^.Vor
          end;
        p := p^.Vor;
        ShowPage(p, 1)
      until (C = #0) or (temp^.Vor = nil);
      NaechsterBefehl := #0;
      Zeile := temp^.Zeile^;
      if C <> #0 then
        ShowPage(p, 1);
    end; (* HochSchieben *)

    procedure TiefSchieben;
    var
      p : TextListe;
    begin
      aktualisiere; p := Top;
      NaechsterBefehl := ^W;
      if p = T then
        exit;
      repeat
        if y < Zeilen then
          y := succ(y)
        else
          begin
            Z := pred(Z);
            temp := temp^.Rueck
          end;
        p := p^.Rueck;
        ShowPage(p, 1)
      until (C = #0) or (p = T);
      NaechsterBefehl := #0;
      Zeile := temp^.Zeile^; if C <> #0 then
        ShowPage(p, 1);
    end; (* TiefSchieben *)

    procedure ZeileVor;
    var
      i : integer;
      p : TextListe;
      cc : char;
    begin
      if temp^.Vor = nil then
        exit;
      NaechsterBefehl := ^x;
      aktualisiere;
      p := Top;
      repeat
        Z := succ(Z);
        if y = Zeilen then
          begin
            ShowPage(temp^.Vor, y);
            p := p^.Vor;
            ShowPage(p, 1)
          end
        else
          begin
            y := succ(y);
            C := #0
          end;
        temp := temp^.Vor; (* Z:=succ(Z); *)
      until (C = #0) or (temp^.Vor = nil);
      NaechsterBefehl := #0;
      Zeile := temp^.Zeile^;
      if C <> #0 then
        ShowPage(p, 1)
    end; (* ZeileVor *)

    procedure ZeileRueck;
    var
      p : TextListe;
    begin
      if temp^.Rueck = nil then
        exit;
      NaechsterBefehl := ^E;
      aktualisiere;
      p := Top;
      repeat
        Z := pred(Z);
        if y = 1 then
          begin
            p := p^.Rueck;
            ShowPage(p, 1)
          end
        else
          begin
            y := pred(y);
            C := #0
          end;
        temp := temp^.Rueck; (* Z:=pred(Z); *)
      until (C = #0) or (temp^.Rueck = nil);
      NaechsterBefehl := #0;
      Zeile := temp^.Zeile^;
      if C <> #0 then
        ShowPage(p, 1)
    end; (* ZeileRueck *)

    procedure SeiteRueck;
    var
      i, j : integer;
      p : TextListe;
    begin
      if Z = 1 then
        exit;
      NaechsterBefehl := ^R;
      aktualisiere;
      repeat
        if Z > Zeilen then
          i := Zeilen
        else
          i := Z;
        Z := Z - pred(i);
        for j := 2 to i do
          temp := temp^.Rueck;
        if Z < y then
          y := Z;
        ShowPage(Top, 1)
      until C <> ^R;
      NaechsterBefehl := #0;
      Zeile := temp^.Zeile^
    end; (* SeiteRueck *)

    procedure SeiteVor;
    var
      i, j : integer;
      p : TextListe;
    begin
      if temp^.Vor = nil then
        exit;
      NaechsterBefehl := ^C;
      aktualisiere;
      repeat
        i := 0;
        while (i < pred(Zeilen)) and (temp^.Vor <> nil) do
          begin
            temp := temp^.Vor;
            i := succ(i);
          end;
        Z := Z + i;
        if temp^.Vor = nil then
          y := 1;
        ShowPage(Top, 1)
      until C <> ^C;
      NaechsterBefehl := #0;
      Zeile := temp^.Zeile^
    end; (* SeiteVor *)

    procedure WortVor;
    begin
      if x > l then
        if temp^.Vor = nil then
          exit
        else
          begin
            ZeileVor;
            x := 1;
            if Zeile[x] in Buchstaben then
              exit
          end;
      while (x <= l) and (Zeile[x] in Buchstaben) do
        x := succ(x);
      while (x <= l) and not(Zeile[x] in Buchstaben) do
        x := succ(x);
    end; (* WortVor *)

    procedure WortRueck;
    begin
      if x > l then
        x := l + 1;
      x := pred(x);
      while (x > 0) and not(Zeile[x] in Buchstaben) do
        x := pred(x);
      if x = 0 then
        begin
          if temp^.Rueck <> nil then
            begin
              ZeileRueck;
              x := l
            end
        end
      else
        while (x > 0) and (Zeile[x] in Buchstaben) do
          x := pred(x);
      x := succ(x)
    end; (* WortRueck *)

    procedure AppendNextLine;
    var
      p : TextListe;
    begin
      if temp^.Vor = nil then
        exit; (* Es gibt keine naechste Zeile *)
      p := temp^.Vor;
      if l + length(p^.Zeile^) < 256 then
        Zeile := Zeile + p^.Zeile^
      else
        begin
          write(^G);
          exit
        end;
      ShowLine(Zeile, x, y, Z);
      DeleteLine(p);
      p := temp^.Vor;
      ShowPage(p, y + 1)
    end; (* AppendNextLine *)

    procedure InsChar;
    begin
      if (l >= MaxSpalten) or (x >= MaxSpalten) then
        begin
          write(^G);
          exit
        end;
      while l < pred(x) do
      Zeile := Zeile + ' ';
      if InsertMode then
        begin
          move(Zeile[x], Zeile[succ(x)], succ(l - x));
          l := succ(l)
        end
      else if l < x then
        l := x;
      Zeile[x] := C;
      x := succ(x);
      if InsertMode then
        AktualisiereBlocks(1, 0);
      ShowLine(Zeile, pred(x), y, Z);
    end; (* InsChar *)

    procedure DeleteChar;
    begin
      delete(Zeile, x, 1);
      AktualisiereBlocks(- 1, 0);
      ShowLine(Zeile, x, y, Z);
    end; (* DeleteChar *)

    procedure LoeschZeichenLinks;
    begin
      if x > 1 then
        begin
          x := pred(x);
          if x <= l then
            DeleteChar
        end
      else if Z > 1 then
        begin
          ZeileRueck;
          x := 0;
          AktualisiereBlocks(0, - 1);
          AktualisiereBlocks(l, 0);
          x := l + 1;
          AppendNextLine;
          aktualisiere;
          ShowPage(temp, y)
        end;
    end; (* LoeschZeichenLinks *)

    procedure LoeschZeichen;
    begin
      if x <= l then
        DeleteChar
      else
        begin
          while l < x do
            Zeile := Zeile + ' ';
          AppendNextLine;
          aktualisiere;
          AktualisiereBlocks(0, - 1);
          ShowPage(temp, y)
        end;
    end; (* LoeschZeichen *)

    procedure LoeschWort;
    var i : integer;
    begin
      if x > l then
        begin
          while l < pred(x) do
            Zeile := Zeile + ' ';
          AppendNextLine;
          aktualisiere;
          AktualisiereBlocks(0, -1);
          ShowPage(temp, y);
          exit;
        end;
      i := x;
      if Zeile[i] in Buchstaben then
        while (i <= l) and (Zeile[i] in Buchstaben) do
          i := succ(i)
      else
        i := succ(i);
      while (i <= l) and (Zeile[i] = ' ') do
        i := succ(i);
      delete(Zeile, x, i - x);
      AktualisiereBlocks(x - i, 0);
      ShowLine(Zeile, x, y, Z)
    end; (* LoeschWort *)

    procedure ZeileLoeschen;
    var
      p, q : TextListe;
      i : integer;
    begin
      if temp^.Vor = nil then
        begin
          Zeile := '';
          aktualisiere;
          ShowLine(Zeile, x, y, Z)
        end
      else
        begin
          NaechsterBefehl := ^y;
          AktualisiereBlocks(0, - 1);
          while (C = ^y) and (temp^.Vor <> nil) do
            begin
              p := temp;
              q := temp^.Vor;
              DeleteLine(temp);
              temp := q;
              ShowPage(temp, y);
              if T = p then
                T := q
            end;
          NaechsterBefehl := #0;
          ShowPage(Top, 1)
        end;
      Zeile := temp^.Zeile^;
      x := 1
    end; (* ZeileLoeschen *)

    procedure ZeileEinfuegen(Anzeige : boolean);
    var
      D : DynString;
      p : TextListe;
    begin
      repeat
        MakeDynString(copy(Zeile, x, succ(l - x)), D);
        Zeile := copy(Zeile, 1, pred(x));
        p := temp;
        AppendLine(p, D);
        aktualisiere;
        with BlockAnfang do
          if Markiert and (Z = tz) and (x < tx) then
            begin
              Z := pred(Z);
              AktualisiereBlocks(0, 1);
              Z := succ(Z);
              tx := tx - pred(x);
              if BlockEnde.Markiert and (BlockEnde.tz = tz) then
                BlockEnde.tx := BlockEnde.tx - pred(x);
            end
          else
            AktualisiereBlocks(0, 1);
        if Anzeige then
          ShowPage(temp, y)
      until (C = #0) or not Anzeige;
      NaechsterBefehl := #0;
    end; (* ZeileEinfuegen *)

    procedure NaechsteZeile;
    var
      p, N : integer;
    begin
      if InsertMode then
        begin
          ZeileEinfuegen(false);
          ShowLine(Zeile, x, y, Z)
        end;
      ZeileVor;
      if IndentMode and InsertMode then
        begin
          p := 1;
          N := DynStringLength(temp^.Rueck^.Zeile);
          if x < N then
            N := pred(x);
          while (p < N) and (temp^.Rueck^.Zeile^[p] = ' ') do
            p := succ(p);
          if p > x then x := 1
          else
            begin
              with BlockAnfang do
                if Markiert and (Z = tz) then
                  begin
                    tx := tx + pred(p);
                    if BlockEnde.Markiert and (BlockEnde.tz = tz) then
                      BlockEnde.tx := BlockEnde.tx + pred(p);
                  end;
              x := p;
              Zeile := Copies(' ', pred(p)) + Zeile;
              aktualisiere;
            end;
        end
      else
        x := 1;
      ShowPage(temp, y);
    end; (* NaechsteZeile *)

    procedure InitEditor;
    var
      i : integer;
    begin
      LR := 0;
      HighVideo;
      if Status then
        begin
          RO := 1;
          Zeilen := ZeilenGanz - 2
        end
      else
        begin
          RO := 0;
          Zeilen := ZeilenGanz - 1
        end;
      if T = nil then
        begin
          new(temp);
          T := temp;
          with temp^ do
            begin
              Vor := nil;
              Rueck := nil;
              MakeDynString('', Zeile)
            end;
          x := 1;
          y := 1;
          Z := 1
        end
      else
        begin
          if x >= Spalten then
            LR := succ(x - Spalten);
          temp := T;
          i := 1;
          while (temp^.Vor <> nil) and (i < Z) do
            begin
              i := succ(i);
              temp := temp^.Vor
            end;
          if i < Z then
            Z := i;
          if y > Z then
            y := Z;
          Anfang := temp;
          for i := 2 to y do
            Anfang := Anfang^.Rueck;
          ShowPage(Anfang, 1)
        end;
      Zeile := temp^.Zeile^;
      if Status then StatusZeile
    end; (* InitEditor *)

(* $I EDT-Q.INC    *)  (* Das Ctrl-Q-Men                       *)
(****************************************************************************)
(*          CTRLQ.INC  Das Ctrl-Q-Men fr den Turbo-Editor TED             *)
(****************************************************************************)
    procedure CtrlQ_Menue(RepeatOp : boolean);

    type Str_32 = string[32];

    const FindStr : Str_32 = ''; ReplaceString : Str_32 = '';
      OptionString : Str_32 = ''; Anzeige : boolean = true;
      Options : set of (upper, words, back, global, norequest) = [];
      LastOp : (FindOp, ReplaceOp) = FindOp;

    var TopList : TextListe;

      function BDynStringPos(A, B : DynString; N : integer; upper : boolean) : integer;
      var p, q : integer;
      begin
        p := 1; q := 0; if N = 0 then 
          begin
            BDynStringPos := 0; exit 
          end;
        repeat
          p := DynStringPos(A, B, p, upper);
          if p > 0
          then if pred(p + DynStringLength(A)) <= N
            then 
              begin
                q := p; p := succ(p)
              end
            else p := 0
        until p = 0;
        BDynStringPos := q
      end; (* BDynStringPos *)

      procedure GetString(var S : Str_32);
      var C : char; l : byte absolute S;
      begin
        LiesZeichen(C);
        if C = ^m then 
          begin
            S := ''; exit
          end
        else if C <> ^R then S := C;
        write(S);
        repeat
          LiesZeichen(C);
          case C of
            #32..#255 : 
              begin
                S := S + C; write(C)
              end;
            ^H : if l > 0 then 
                   begin
                     l := pred(l); write(^H' '^H)
                   end
          end
        until (C = ^m) or (l >= 32)
      end; (* GetString *)

      procedure ZumAnfang;
      begin
        aktualisiere; temp := T; Zeile := temp^.Zeile^;
        x := 1; y := 1; Z := 1; LR := 0; ShowPage(T, 1)
      end; (* ZumAnfang *)

      procedure ZumEnde;
      var p : TextListe;
      begin
        aktualisiere; p := temp;
        while p^.Vor <> nil do
          begin
            p := p^.Vor; Z := succ(Z); if y < Zeilen then y := succ(y)
          end;
        temp := p; Zeile := temp^.Zeile^; x := succ(l);
        if x > Spalten + LR then LR := x - Spalten;
        ShowPage(Top, 1)
      end; (* ZumEnde *)

      procedure SeitenAnfang;
      begin
        aktualisiere; temp := Top; Zeile := temp^.Zeile^; Z := succ(Z - y); y := 1;
      end; (* SeitenAnfang *)

      procedure SeitenEnde;
      begin
        aktualisiere;
        while (temp^.Vor <> nil) and (y < Zeilen) do 
          begin
            temp := temp^.Vor; y := succ(y); Z := succ(Z)
          end;
        Zeile := temp^.Zeile^
      end; (* SeitenEnde *)

      procedure RetteZeile;
      begin
        Zeile := temp^.Zeile^; ShowLine(Zeile, succ(LR), y, Z);
        if x > l then x := l + 1;
      end; (* RetteZeile *)

      procedure LoescheBisZeilenEnde;
      var N : integer;
      begin
        N := 1 + l - x;
        if N > 0 then 
          begin
            l := pred(x); AktualisiereBlocks(- N, 0); ShowLine(Zeile, x, y, Z)
          end;
      end;

      procedure LiesOptions;
      var i : integer;
      begin
        gotoxy(1, 1); write('Optionen: '); clreol; GetString(OptionString);
        Options := [];
        for i := 1 to length(OptionString) do
          case upcase(OptionString[i]) of
            'G' : Options := Options + [global];
            'N' : Options := Options + [norequest];
            'U' : Options := Options + [upper];
            'W' : Options := Options + [words];
            'B' : Options := Options + [back];
          end (* CASE *)
      end; (* LiesOptions *)

      function Found : boolean;
      var p : TextListe; N, tz, ty, tx : integer; gef : boolean;
        D : DynString;

        procedure CheckWord;
        begin
          gef := N > 0;
          if gef and (words in Options) then
            gef := (((N = 1) or not(p^.Zeile^[pred(N)] in Buchstaben)) and
                    ((N + length(FindStr) > DynStringLength(p^.Zeile)) or
                     not(p^.Zeile^[N + length(FindStr)] in Buchstaben)))
        end; (* CheckWord *)

      begin
        aktualisiere; p := temp; tx := x; ty := y; tz := Z; N := tx;
        if back in Options then N := pred(N); MakeDynString(FindStr, D);
        repeat
          if back in Options then 
            begin
              N := BDynStringPos(D, p^.Zeile, N, upper in Options); CheckWord;
              if N = 0 then 
                begin
                  p := p^.Rueck; N := 255; tz := pred(tz); if ty > 1 then ty := pred(ty)
                end 
              else if not gef then N := pred(N)
            end 
          else 
            begin
              N := DynStringPos(D, p^.Zeile, N, upper in Options); CheckWord;
              if N = 0 then
                begin
                  N := 1; p := p^.Vor; tz := succ(tz); ty := succ(ty) 
                end
              else if not gef then N := succ(N)
            end;
        until (p = nil) or gef; ForgetDynString(D);
        if gef then 
          begin
            x := N; if ty > Zeilen then y := Zeilen 
            else y := ty;
            temp := p; Z := tz; Zeile := temp^.Zeile^;
            if Anzeige then ShowPage(Top, 1)
          end;
        Found := gef
      end; (* Found *)

      procedure Suche(Neu : boolean);
      label Ausgang;
      begin
        if Neu then 
          begin
            gotoxy(1, 1); write('Suchen: '); clreol; GetString(FindStr);
            if FindStr = '' then goto Ausgang;
            LiesOptions;
          end;
        if not Found then
          begin
            gotoxy(1, 1); write('Suchbegriff nicht gefunden <ESC> '); clreol;
            repeat
              LiesZeichen(C)
            until C = ^[; C := #0
          end
        else if not(back in Options) then x := x + length(FindStr);
Ausgang:
        if Status then StatusZeile
        else 
          begin
            TopList := Top; ShowLine(TopList^.Zeile^, 1 + LR, 1, 1 + Z - y) 
          end
      end; (* Suche *)

      procedure Replace(Neu : boolean);
      label Ausgang;
      const Schranke = 1000; (* Ausprobieren! *)
      var Erfolg, Oben : boolean; C : char; Count : integer;
      begin
        if Neu then 
          begin
            gotoxy(1, 1); write('Suchen: '); clreol;
            GetString(FindStr); if FindStr = '' then goto Ausgang;
            gotoxy(1, 1); write('Ersetzen durch: '); clreol;
            GetString(ReplaceString);
            LiesOptions;
          end; (* IF Neu *)
        if global in Options then
          if back in Options then ZumEnde 
          else ZumAnfang;
        Erfolg := Found; Anzeige := true;
        if not Erfolg then 
          begin
            gotoxy(1, 1); write('Suchbegriff nicht gefunden <ESC> '); clreol;
            repeat
              LiesZeichen(C)
            until C = ^[; C := #0
          end 
        else 
          repeat
            if x > Spalten + LR then 
              begin
                aktualisiere; LR := x - Spalten; ShowPage(Top, 1) 
              end 
            else
              if x <= LR then 
                begin
                  aktualisiere; LR := pred(x); ShowPage(Top, 1) 
                end;
            if not(norequest in Options) then 
              begin
                gotoxy(1, 1); write('Ersetzen (J/N)? '); clreol;
                Oben := true; Count := 0; C := #0;
                repeat
                  if Count = Schranke then 
                    begin
                      if Oben then gotoxy(17, 1)
                      else gotoxy(x - LR, y + RO);
                      Count := 0; Oben := not Oben;
                    end;
                  if KeyPressed then 
                    begin
                      LiesZeichen(C); C := upcase(C)
                    end;
                  Count := Count + 1
                until (C = 'J') or (C = 'N') or (C = ^U);
                if C = ^U then goto Ausgang;
              end 
            else if KeyPressed then
              begin
                LiesZeichen(C); Anzeige := false 
              end;
            if (C = 'J') or (norequest in Options) then
              begin
                delete(Zeile, x, length(FindStr));
                insert(ReplaceString, Zeile, x);
                if Anzeige then ShowLine(Zeile, succ(LR), y, Z) 
              end;
            if not(back in Options) then x := x + length(ReplaceString);
            if global in Options then Erfolg := Found 
            else Erfolg := false
          until not Erfolg;
        if global in Options then
          if back in Options then ZumAnfang 
          else ZumEnde;
Ausgang: Anzeige := true;
        if Status then StatusZeile
        else 
          begin
            TopList := Top; ShowLine(TopList^.Zeile^, 1 + LR, 1, 1 + Z - y) 
          end
      end; (* Replace *)

      procedure RepeatLastOp;  (* Wird durch ^L gestartet *)
      begin
        case LastOp of
          FindOp : Suche(false);
          ReplaceOp : Replace(false)
        end; (* CASE *)
      end; (* RepeatLastOp *)

    begin
      if RepeatOp then 
        begin
          RepeatLastOp; exit 
        end;
      if Status then 
        begin
          gotoxy(1, 1); write('^Q')
        end;
      LiesZeichen(C);
      case C of
        ^R, 'r', 'R' : ZumAnfang;
        ^C, 'c', 'C' : ZumEnde;
        ^E, 'e', 'E' : SeitenAnfang;
        ^x, 'x', 'X' : SeitenEnde;
        ^S, 's', 'S' : x := 1;
        ^D, 'd', 'D' : x := l + 1;
        ^l, 'l', 'L' : RetteZeile;
        ^y, 'y', 'Y' : LoescheBisZeilenEnde;
        ^i, 'i', 'I' : ChangeIndent;
        ^f, 'f', 'F' :
          begin
            Suche(true); LastOp := FindOp 
          end;
        ^A, 'a', 'A' :
          begin
            Replace(true); LastOp := ReplaceOp
          end;
      end; (* CASE *)
      if Status then 
        begin
          gotoxy(1, 1); write('  ')
        end;
    end; (* CtrlQ_Menue *)
(* $I EDT-K.INC    *)  (* Das Ctrl-K-Men                       *)
(****************************************************************************)
(*CTRLK.INC  Das Ctrl-K-Menue fuer den Turbo-Editor TED mit Blockfunktionen *)
(****************************************************************************)
    procedure CtrlK_Menue;

    type Str40 = string[40];

    var TopListe : TextListe;
      Abbruch : boolean;

    const Anzeige : boolean = true;

      procedure ErsteZeile;
      begin
        if Status then StatusZeile
        else 
          begin
            TopListe := Top; ShowLine(TopListe^.Zeile^, 1 + LR, 1, 1 + Z - y) 
          end
      end; (* ErsteZeile *)

      function LegalBlock : boolean;
      begin
        LegalBlock := ((BlockAnfang.tz < BlockEnde.tz) or
                       (((BlockAnfang.tz = BlockEnde.tz) or SpaltenBlock) and
                        (BlockAnfang.tx < BlockEnde.tx))) and
        (BlockAnfang.Markiert and BlockEnde.Markiert);
      end; (* LegalBlock *)

      function CursorInBlock : boolean;
      var h : boolean;
      begin
        h := (Z >= BlockAnfang.tz) and (Z <= BlockEnde.tz);
        if SpaltenBlock then h := h and (x >= BlockAnfang.tx) and (x < BlockEnde.tx)
        else
          begin
            if (Z = BlockAnfang.tz) and (x < BlockAnfang.tx) then h := false 
            else
              if (Z = BlockEnde.tz) and (x >= BlockEnde.tx) then h := false;
          end;
        CursorInBlock := h;
      end; (* CursorInBlock *)

      function CursorAuf(Block : BlockMarke) : boolean;
      begin
        CursorAuf := (Z = Block.tz) and (x = Block.tx)
      end; (*CursorAuf *)

      procedure BlockWegmachen;
      begin
        BlockAnfang.Markiert := false; BlockEnde.Markiert := false;
        aktualisiere; ShowPage(Top, 1)
      end; (* BlockWegmachen *)

      procedure BlockMarkieren(var B : BlockMarke; Anzeige : boolean);
      begin
        B.tx := x; B.tz := Z; B.Markiert := true;
        if BlockAnfang.Markiert and BlockEnde.Markiert and Anzeige then
          begin
            aktualisiere; ShowPage(Top, 1) 
          end;
      end; (* BlockMarkieren *)

      procedure WortMarkieren;
      var Xsave : integer;
      begin
        Xsave := x;
        if x > l then x := l + 1;
        while (x > 0) and (Zeile[x] in Buchstaben) do x := pred(x);
        x := succ(x); BlockMarkieren(BlockAnfang, false);
        while (x <= l) and (Zeile[x] in Buchstaben) do x := succ(x);
        BlockMarkieren(BlockEnde, false);
        x := Xsave; aktualisiere; ShowPage(Top, 1)
      end; (* WortMarkieren *)

      procedure ZeileMarkieren;
      begin
        SpaltenBlock := false; x := 1; BlockMarkieren(BlockAnfang, false);
        ZeileVor; BlockMarkieren(BlockEnde, true);
      end; (*ZeileMarkieren *)

      procedure RestauriereBlock;
      var p : TextListe; i : integer;
      begin
        p := T;
        for i := 1 to pred(BlockAnfang.tz) do p := p^.Vor;
        BlockAnfang.Line := p;
        for i := BlockAnfang.tz to pred(BlockEnde.tz) do p := p^.Vor;
        BlockEnde.Line := p;
      end; (*RestauriereBlock *)

      procedure GetPosition(m : Str40; var B : BlockMarke);
      var C : char;
      begin
        LowVideo; gotoxy(1, 1); write('  ', m, ' <RETURN>'); clreol; HighVideo;
        repeat
          if x > Spalten + LR then
            begin
              aktualisiere; LR := x - Spalten; ShowPage(Top, 1) 
            end 
          else
            if x <= LR then 
              begin
                aktualisiere; LR := pred(x); ShowPage(Top, 1) 
              end;
          gotoxy(x - LR, y + RO); LiesZeichen(C);
          case C of
            ^S : if x > 1 then x := pred(x);
            ^D : if x < MaxSpalten then x := succ(x);
            ^x : ZeileVor;
            ^E : ZeileRueck;
            ^R : SeiteRueck;
            ^C : SeiteVor;
            ^f : WortVor;
            ^A : WortRueck;
            ^W : TiefSchieben;
            ^Z : HochSchieben;
            ^q : CtrlQ_Menue(false);
          end;
        until (C = ^m) or (C = ^U); Abbruch := (C = ^U);
        if not Abbruch then BlockMarkieren(B, true)
      end; (* GetPosition *)

      procedure LiesPositionen(Anzahl : integer);
      var EinfuegePosition : BlockMarke;
      begin
        Abbruch := false;
        if BlockAnfang.Markiert and BlockEnde.Markiert then exit;
        GetPosition('Blockanfang', BlockAnfang); if Abbruch then exit;
        GetPosition('Blockende', BlockEnde); if Abbruch then exit;
        if Anzahl = 3 then GetPosition('Neue Position', EinfuegePosition);
      end; (* LiesPosition *)

      procedure DelBlock;
      var D : DynString; Block : TextListe; N, i : integer;
      begin
        N := BlockEnde.tx - BlockAnfang.tx;
        with BlockAnfang do
          if tz = BlockEnde.tz then
            begin
              if Z = tz then delete(Zeile, tx, N) 
              else delete(Line^.Zeile^, tx, N);
            end
          else if SpaltenBlock then
            for i := tz to BlockEnde.tz do
              begin
                if Z = i then delete(Zeile, tx, N) 
                else delete(Line^.Zeile^, tx, N);
                Line := Line^.Vor;
              end
          else 
            begin
              MakeDynString(copy(Line^.Zeile^, 1, pred(tx)) +
                            copy(BlockEnde.Line^.Zeile^, BlockEnde.tx, 255), D);
              ForgetDynString(Line^.Zeile); Line^.Zeile := D;
              Block := Line^.Vor; Line^.Vor := BlockEnde.Line^.Vor;
              if Line^.Vor <> nil then Line^.Vor^.Rueck := Line;
              if BlockEnde.tz = Z then
                begin
                  Zeile := Line^.Zeile^; temp := Line 
                end;
              BlockEnde.Line^.Vor := nil; LoescheText(Block);
            end;
      end; (* DelBlock *)

      procedure InsBlock(TL : TextListe);
      var D : DynString; p : TextListe; S : StatString; dx, i : integer;
      begin
        if SpaltenBlock and (BlockAnfang.tz <> BlockEnde.tz) then
          begin
            dx := BlockEnde.tx - BlockAnfang.tx;
            for i := BlockAnfang.tz to BlockEnde.tz do
              begin
                if l < x then Zeile := Zeile + Copies(' ', pred(x) - l);
                S := copy(TL^.Zeile^, BlockAnfang.tx, dx);
                if length(S) < dx then S := S + Copies(' ', dx - length(S));
                insert(S, Zeile, x);
                aktualisiere; TL := TL^.Vor;
                if (temp^.Vor = nil) and (TL <> nil) then
                  begin
                    MakeDynString('', D); Zeile := '';
                    AppendLine(temp, D); aktualisiere 
                  end
                else if TL <> nil then
                  begin
                    temp := temp^.Vor; Zeile := temp^.Zeile^ 
                  end;
              end;
            for i := 1 to BlockEnde.tz - BlockAnfang.tz do temp := temp^.Rueck;
            Zeile := temp^.Zeile^; aktualisiere
          end
        else 
          begin
            p := temp^.Vor; temp^.Vor := TL; TL^.Rueck := temp;
            while TL^.Vor <> nil do TL := TL^.Vor;
            TL^.Vor := p; if p <> nil then p^.Rueck := TL;
            MakeDynString(TL^.Zeile^ + copy(Zeile, x, 255), D);
            ForgetDynString(TL^.Zeile); TL^.Zeile := D;
            l := pred(x); AppendNextLine; aktualisiere;
          end
      end; (* InsBlock *)

      procedure MakeBlock(var TL : TextListe);
      var p, q : TextListe; D : DynString; A : integer;
      begin
        NeuerText(TL);
        if BlockAnfang.tz = BlockEnde.tz then 
          begin
            MakeDynString(copy(BlockAnfang.Line^.Zeile^, BlockAnfang.tx,
                               BlockEnde.tx - BlockAnfang.tx), D);
            AppendLine(TL, D)
          end
        else 
          begin
            if SpaltenBlock then A := 1
            else A := BlockAnfang.tx;
            MakeDynString(copy(BlockAnfang.Line^.Zeile^, A, 255), D);
            AppendLine(TL, D); p := TL; q := BlockAnfang.Line^.Vor;
            while (q <> BlockEnde.Line) and (HeapAvail > 1000) do
              begin
                MakeDynString(q^.Zeile^, D); AppendLine(p, D); q := q^.Vor 
              end;
            MakeDynString(copy(BlockEnde.Line^.Zeile^, 1, pred(BlockEnde.tx)), D);
            AppendLine(p, D)
          end;
      end; (* MakeBlock *)

      procedure BlockSchreiben;
      label Ausgang;
      var p : TextListe;
        f : text;
        fn : string[32];
        ta, N : integer;
      begin
        LiesPositionen(2);
        if not Abbruch and LegalBlock then
          begin
            gotoxy(1, 1); write('Datei zum Schreiben: '); clreol;
            readln(fn); if fn = '' then goto Ausgang;
            assign(f, fn); rewrite(f);
            aktualisiere; RestauriereBlock;
            if BlockAnfang.tz = BlockEnde.tz
            then with BlockAnfang do
              write(f, copy(Line^.Zeile^, tx, BlockEnde.tx - tx))
            else 
              begin
                if SpaltenBlock
                then 
                  begin
                    ta := BlockAnfang.tx; N := BlockEnde.tx - ta 
                  end
                else 
                  begin
                    ta := 1; N := 255 
                  end;
                with BlockAnfang do writeln(f, copy(Line^.Zeile^, tx, N));
                p := BlockAnfang.Line^.Vor;
                while p <> BlockEnde.Line do 
                  begin
                    writeln(f, copy(p^.Zeile^, ta, N)); p := p^.Vor
                  end;
                with BlockEnde do
                  if tx = 1 then writeln(f)
                  else write(f, copy(Line^.Zeile^, ta, tx - ta));
              end;
            close(f)
          end;
Ausgang: ErsteZeile
      end; (* BlockSchreiben *)

      procedure BlockLesen;
      label Ausgang;
      var f : text;
        fn : string[32];
        TL, p : TextListe;
        N : integer;
      begin
        repeat
          gotoxy(1, 1); write('Datei zum Lesen: '); clreol;
          readln(fn); if fn = '' then goto Ausgang;
          (*$I-*) assign(f, fn); reset(f) (*$I+*) ;
        until ioresult = 0; close(f);
        BlockMarkieren(BlockAnfang, false);
        if l < x then Zeile := Zeile + Copies(' ', x - l);
        LiesText(f, TL); p := TL;
        if p = nil then N := 0
        else 
          begin
            N := 1; while p^.Vor <> nil do 
              begin
                N := succ(N); p := p^.Vor 
              end;
          end;
        with BlockEnde do
          begin
            tz := BlockAnfang.tz + pred(N); tx := succ(DynStringLength(p^.Zeile));
            if BlockAnfang.tz = tz then tx := pred(tx + BlockAnfang.tx);
            Markiert := true;
          end;
        SpaltenBlock := false; InsBlock(TL); ShowPage(Top, 1);
Ausgang: ErsteZeile
      end; (* BlockLesen *)

      procedure BlockLoeschen;
      begin
        LiesPositionen(2);
        if not Abbruch and LegalBlock then with BlockAnfang do
          begin
            aktualisiere; RestauriereBlock; DelBlock;
            if SpaltenBlock or (tz = BlockEnde.tz) then aktualisiere;
            if CursorInBlock or (CursorAuf(BlockEnde) and not SpaltenBlock) then
              begin
                Z := tz; x := tx; y := y - Z + tz; if y < 1 then y := 1 
              end
            else if not SpaltenBlock and (Z > tz) then
              begin
                if Z - BlockEnde.tz < y then y := y - BlockEnde.tz + tz;
                if y < 1 then y := y - BlockEnde.tz; Z := Z - BlockEnde.tz + tz
              end;
            Markiert := false; BlockEnde.Markiert := false;
            InitEditor
          end;
        ErsteZeile
      end; (* BlockLoeschen *)

      procedure BlockKopieren;
      var Block : TextListe;
      begin
        LiesPositionen(3);
        if not Abbruch and LegalBlock and
        (not CursorInBlock or CursorAuf(BlockAnfang)) then
          begin
            aktualisiere; RestauriereBlock;
            if l < x then Zeile := Zeile + Copies(' ', x - l);
            MakeBlock(Block); InsBlock(Block);
            with BlockEnde do
              begin
                if tz = BlockAnfang.tz
                then
                  begin
                    tz := Z; tx := x + tx - BlockAnfang.tx
                  end
                else 
                  begin
                    tz := Z + tz - BlockAnfang.tz;
                    if SpaltenBlock then tx := x + tx - BlockAnfang.tx 
                  end;
              end;
            BlockAnfang.tx := x; BlockAnfang.tz := Z;
            ShowPage(Top, 1);
          end;
        ErsteZeile
      end; (* BlockKopieren *)

      procedure BlockVerschieben;
      var Block : TextListe; D : DynString; dx, dz : integer; h : boolean;
      begin
        LiesPositionen(3);
        if Abbruch or not LegalBlock or
        (CursorInBlock and not(SpaltenBlock and (BlockAnfang.tx = x))) or
        CursorAuf(BlockEnde) then 
          begin
            ErsteZeile; exit 
          end;
        aktualisiere; RestauriereBlock;
        if l < x then Zeile := Zeile + Copies(' ', x - l);
        with BlockAnfang do
          begin
            dx := BlockEnde.tx - tx; dz := BlockEnde.tz - tz;
            MakeBlock(Block); DelBlock;
            h := SpaltenBlock or (tz = BlockEnde.tz);
            if h and (Z >= tz) and (Z <= BlockEnde.tz) and (x > tx) then x := x - dx;
            InsBlock(Block);
            if h then BlockEnde.tx := x + dx
            else if BlockEnde.tz <= Z then
              begin
                if Z - BlockEnde.tz < y then
                  begin
                    if (Z - tz < y) and (y > dz) then y := y - dz
                    else if Z - dz < y then y := Z - dz;
                  end;
                Z := Z - dz;
              end;
            tx := x; tz := Z; BlockEnde.tz := tz + dz; InitEditor
          end;
      end; (* BlockVerschieben *)

    begin
      if Status then 
        begin
          gotoxy(1, 1); write('^K') 
        end;
      LiesZeichen(C);
      case C of
        ^D, 'd', 'D' : C := ^[;
        ^x, 'x', 'X' : C := ^[;
        ^S, 's', 'S' : SchreibText(f, T);
        ^y, 'y', 'Y' : BlockLoeschen;
        ^R, 'r', 'R' : BlockLesen;
        ^W, 'w', 'W' : BlockSchreiben;
        ^C, 'c', 'C' : BlockKopieren;
        ^V, 'v', 'V' : BlockVerschieben;
        ^B, 'b', 'B' : BlockMarkieren(BlockAnfang, true);
        ^K, 'k', 'K' : BlockMarkieren(BlockEnde, true);
        ^h, 'h', 'H' : BlockWegmachen;
        ^l, 'l', 'L' : ZeileMarkieren;
        ^T, 't', 'T' : WortMarkieren;
        ^N, 'n', 'N' : 
          begin
            SpaltenBlock := not SpaltenBlock;
            aktualisiere; ShowPage(Top, 1) 
          end;
      end;
      if Status then 
        begin
          gotoxy(1, 1); write('  ') 
        end;
    end; (* CtrlK_Menue *)

(****************************************************************************)
(*  TED-3.INC   Zweiter Teil der Prozedur EditText fr den Turbo-Editor TED *)
(****************************************************************************)

  begin (* EditText *)
    if Spalten >= MaxX then
      Spalten := pred(MaxX);
    if ZeilenGanz >= MaxY then
      ZeilenGanz := pred(MaxY);
    SetTextColor;
    BlockAnfang.Markiert := false;
    BlockEnde.Markiert := false;
    InitEditor;
    repeat
      if Status then
        SchreibStatus;
      if HeapAvail < 1000 then
        begin
          gotoxy(1, 1);
          write(^G'Speicher voll!')
        end;
      if x > Spalten + LR then
        begin
          aktualisiere;
          LR := x - Spalten;
          ShowPage(Top, 1)
        end
      else if x <= LR then
        begin
          aktualisiere;
          LR := pred(x);
          ShowPage(Top, 1)
        end;
      gotoxy(x - LR, y + RO);
      LiesZeichen(C);
      case C of
        #13 : NaechsteZeile;
        ^S : if x > 1 then
          x := pred(x);
        ^D : if x < MaxSpalten then x := succ(x);
        ^x : ZeileVor;
        ^E : ZeileRueck;
        ^R : SeiteRueck;
        ^C : SeiteVor;
        ^f : WortVor;
        ^A : WortRueck;
        ^q : CtrlQ_Menue(false);
        ^l : CtrlQ_Menue(true);
        ^V : ChangeMode;
        ^i : if x < pred(MaxSpalten - 8) then x := 8 * succ(x div 8);
        #27:;
        else
          edited := true;
        end;
        case C of
        #32..#255 : InsChar;
        ^h : LoeschZeichenLinks;
        ^G : LoeschZeichen;
        ^T : LoeschWort;
        ^y : ZeileLoeschen;
        ^W : TiefSchieben;
        ^Z : HochSchieben;
        ^K : CtrlK_Menue;
        ^N :
          begin
            NaechsterBefehl := ^N; ZeileEinfuegen(true)
          end;
      end; (* CASE *)
    until C = #27;
    aktualisiere;
  end; (* EditText *)


begin
  OpenWindow (EdtX1,EdtY1,EdtX2,EdtY2);
  WindowTop(EdtTop);
  CursorOn;
  if getenv('VIDEO') = 'BIOS' then
    DirectVideo := false;
  CheckBreak := false;
  assign(f, Filename);
  (*$I-*)
  reset(f);
  (*$I+*)
  if ioresult = 0 then
    begin
      close(f);
      LiesText(f, T);
    end
  else
    NeuerText(T);
  x := 1;
  y := 1;
  Z := 1;
  ClrScr;
  repeat
    EditText(T, x, y, Z, 79, 24, true);
    if edited then
    begin
    case alertbox(28, 7, 'Text speichern ?', ' J a ! N e i n ', 1) of
      1 :
        begin
          SchreibText(f, T);
          Filename := '';
        end;
      2 :
        Filename := '';
    end;
    end
    else
      filename := '';
  until Filename = '';
  LoescheText(T);
  CloseWindow;
end;

end.
